package socket

import (
	"crypto/sha256"
	"encoding/binary"
	"encoding/hex"
	"fmt"
	"os/user"
	"strings"
	"unsafe"

	"golang.org/x/sys/windows"
)

const (
	// windows consts
	cryptProtectMemoryBlockSize    = 16
	cryptProtectMemoryCrossProcess = 0x01

	// Pageant consts
	pageantPipeName = `\\.\pipe\pageant.%s.%s`
	pageantName     = "Pageant"
)

var (
	crypt32                = windows.NewLazySystemDLL("crypt32.dll")
	procCryptProtectMemory = crypt32.NewProc("CryptProtectMemory")
)

func NewPageantPipe() (Socket, error) {
	u, err := user.Current()
	if err != nil {
		return nil, fmt.Errorf("can't get current user: %w", err)
	}

	userName := u.Username
	if idx := strings.LastIndexByte(userName, '\\'); idx > 0 {
		userName = userName[idx+1:]
	}

	// reference: https://git.tartarus.org/?p=simon/putty.git;a=blob;f=windows/agent-client.c;h=4eb0bcfb9667ecb6f0f63c1fb8d5ea70651d04ba;hb=HEAD#l128
	pipePath := fmt.Sprintf(pageantPipeName, userName, capiObfuscateString(pageantName))
	return NewPipeSocket(pipePath)
}

// reference: https://git.tartarus.org/?p=simon/putty.git;a=blob;f=windows/utils/cryptoapi.c;h=c3752c59dd13ab6a0848ee02ce1a40934f9bacb2;hb=HEAD#l29
func capiObfuscateString(realname string) string {
	cryptLen := len(realname) + 1
	cryptLen += cryptProtectMemoryBlockSize - 1
	cryptLen /= cryptProtectMemoryBlockSize
	cryptLen *= cryptProtectMemoryBlockSize

	cryptData := make([]byte, cryptLen+4)
	binary.BigEndian.PutUint32(cryptData[:], uint32(cryptLen))
	copy(cryptData[4:], realname)

	pDataIn := uintptr(unsafe.Pointer(&cryptData[4]))
	cbDataIn := uintptr(cryptLen)
	dwFlags := uintptr(cryptProtectMemoryCrossProcess)
	// pageant ignores errors
	_, _, _ = procCryptProtectMemory.Call(pDataIn, cbDataIn, dwFlags)

	hash := sha256.Sum256(cryptData)
	return hex.EncodeToString(hash[:])
}
