package socket

import (
	"context"
	"errors"
	"fmt"
	"net"
	"syscall"

	"github.com/Microsoft/go-winio"
)

var _ Socket = (*PipeSocket)(nil)

type PipeSocket struct {
	pipePath string
	listener net.Listener
	conns    chan net.Conn
	errs     chan error
	done     chan struct{}
	closed   bool
}

func NewPipeSocket(pipePath string) (Socket, error) {
	return &PipeSocket{
		pipePath: pipePath,
		conns:    make(chan net.Conn),
		errs:     make(chan error, 1),
	}, nil
}

func (s *PipeSocket) Addr() string {
	return s.pipePath
}

func (s *PipeSocket) Listen() error {
	if isPipeExists(s.pipePath) {
		return fmt.Errorf("pipe already exists: %s", s.pipePath)
	}

	l, err := winio.ListenPipe(s.pipePath, nil)
	if err != nil {
		return fmt.Errorf("pipe listen failed: %w", err)
	}

	s.listener = l
	s.done = make(chan struct{})

	go func() {
		defer close(s.done)

		for {
			conn, err := s.listener.Accept()
			if err != nil {
				s.errs <- err

				if errors.Is(err, winio.ErrPipeListenerClosed) {
					return
				}
				continue
			}

			s.conns <- conn
		}

	}()
	return nil
}

func (s *PipeSocket) Accept(ctx context.Context) (Conn, error) {
	if s.closed {
		return nil, net.ErrClosed
	}

	select {
	case err := <-s.errs:
		if errors.Is(err, winio.ErrPipeListenerClosed) {
			return nil, net.ErrClosed
		}

		return nil, err
	case conn := <-s.conns:
		creds, err := unixCreds(conn)
		if err != nil {
			_ = conn.Close()
			return nil, fmt.Errorf("unable to get unix creds: %w", err)
		}

		return &connWrapper{
			ReadWriteCloser: conn,
			peer:            creds,
		}, nil
	case <-ctx.Done():
		return nil, ctx.Err()
	}
}

func (s *PipeSocket) Close() error {
	if s.closed {
		return nil
	}

	s.closed = true
	err := s.listener.Close()

	close(s.conns)
	for c := range s.conns {
		_ = c.Close()
	}

	if s.done != nil {
		<-s.done
	}
	close(s.errs)
	return err
}

func isPipeExists(path string) bool {
	namep, _ := syscall.UTF16PtrFromString(path)
	var fd syscall.Win32finddata
	sh, err := syscall.FindFirstFile(namep, &fd)
	if err != nil {
		return false
	}

	_ = syscall.FindClose(sh)
	return true
}
