package socket

import (
	"encoding/json"
	"fmt"

	"gopkg.in/yaml.v2"
)

var _ yaml.Unmarshaler = (*Kind)(nil)
var _ yaml.Marshaler = (*Kind)(nil)
var _ json.Unmarshaler = (*Kind)(nil)
var _ json.Marshaler = (*Kind)(nil)

type Kind uint8

const (
	KindNone Kind = iota
	KindUnix
	KindCygwin
	KindPageant
	KindPageantPipe
	KindPageantWindow
	KindPipe
	KindDummy
)

func (k Kind) String() string {
	switch k {
	case KindNone:
		return "none"
	case KindUnix:
		return "unix"
	case KindCygwin:
		return "cygwin"
	case KindPageant:
		return "pageant"
	case KindPageantPipe:
		return "pageant-pipe"
	case KindPageantWindow:
		return "pageant-window"
	case KindPipe:
		return "pipe"
	case KindDummy:
		return "dummy"
	default:
		return fmt.Sprintf("socket_%d", k)
	}
}

func (k *Kind) fromString(v string) error {
	switch v {
	case "none":
		*k = KindNone
	case "unix":
		*k = KindUnix
	case "cygwin":
		*k = KindCygwin
	case "pageant":
		*k = KindPageant
	case "pageant-pipe", "pageant_pipe":
		// "pageant_pipe" is for backward compatibility
		*k = KindPageantPipe
	case "pageant-window":
		*k = KindPageantWindow
	case "pipe":
		*k = KindPipe
	case "dummy":
		*k = KindDummy
	default:
		return fmt.Errorf("unknown socket: %s", v)
	}

	return nil
}

func (k Kind) MarshalYAML() (interface{}, error) {
	return k.String(), nil
}

func (k *Kind) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var v string
	if err := unmarshal(&v); err != nil {
		return err
	}

	return k.fromString(v)
}

func (k Kind) MarshalJSON() ([]byte, error) {
	return json.Marshal(k.String())
}

func (k *Kind) UnmarshalJSON(in []byte) error {
	var v string
	if err := json.Unmarshal(in, &v); err != nil {
		return err
	}

	return k.fromString(v)
}
