package ui

import (
	"context"
	"fmt"
	"os"
	"os/signal"
	"path/filepath"
	"runtime"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

var _ Notifier = (*ConsoleApp)(nil)
var _ App = (*ConsoleApp)(nil)

type ConsoleApp struct {
	notifier  osNotifier
	iconsPath string
	log       log.Logger
	doneCh    chan struct{}
	ctx       context.Context
	cancelFn  context.CancelFunc
}

func NewConsoleApp(opts ...Option) (*ConsoleApp, error) {
	notifier, err := newNotifier()
	if err != nil {
		return nil, fmt.Errorf("can't create notifier: %w", err)
	}

	ctx, cancel := context.WithCancel(context.Background())
	out := &ConsoleApp{
		notifier: notifier,
		log:      &nop.Logger{},
		ctx:      ctx,
		cancelFn: cancel,
	}

	for _, opt := range opts {
		switch v := opt.(type) {
		case optionLogger:
			out.log = v.log
		case optionIconsPath:
			out.iconsPath = v.iconsPath
		}
	}

	return out, nil
}

func (a *ConsoleApp) Start(mainFn MainFn) error {
	return a.start(context.Background(), mainFn)
}

func (a *ConsoleApp) start(ctx context.Context, mainFn MainFn) error {
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()

	errChan := make(chan error, 1)
	a.doneCh = make(chan struct{})
	go func() {
		errChan <- mainFn(ctx)
		close(a.doneCh)
	}()

	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM)

	select {
	case <-a.ctx.Done():
		signal.Stop(sigChan)
		cancel()
		<-a.doneCh
		return nil
	case <-sigChan:
		cancel()
		<-a.doneCh
		return nil
	case err := <-errChan:
		return err
	}
}

func (a *ConsoleApp) Shutdown(ctx context.Context) error {
	a.cancelFn()
	defer func() {
		if err := a.notifier.Close(); err != nil {
			a.log.Error("failed to close notifier", log.Error(err))
		}
	}()

	if a.doneCh == nil {
		return nil
	}

	select {
	case <-ctx.Done():
		return ctx.Err()
	case <-a.doneCh:
		return nil
	}
}

func (a *ConsoleApp) Notify(kind NotificationKind, text string) (*Notification, error) {
	notif, err := a.notifier.Notify(
		AppName,
		text,
		a.iconPath(kind),
		a.urgency(kind),
	)
	if err != nil {
		return nil, err
	}

	return &Notification{
		n: notif,
	}, nil
}

func (a *ConsoleApp) NotifyAndForget(kind NotificationKind, text string, ttl time.Duration) error {
	notif, err := a.notifier.Notify(
		AppName,
		text,
		a.iconPath(kind),
		a.urgency(kind),
	)
	if err != nil {
		return err
	}

	go func() {
		time.Sleep(ttl)
		notif.Close()
	}()

	return nil
}

func (a *ConsoleApp) ScheduleNotify(postpone time.Duration, kind NotificationKind, text string) (*Notification, error) {
	timer := time.NewTimer(postpone)

	var notification Notification
	var ctx context.Context
	ctx, notification.cancelCtx = context.WithCancel(context.Background())

	go func() {
		select {
		case <-ctx.Done():
			timer.Stop()
		case <-timer.C:
			notification.mu.Lock()
			if nn, err := a.Notify(kind, text); err != nil {
				a.log.Warn("couldn't send touch notification", log.Error(err))
			} else {
				notification.n = nn
			}
			notification.mu.Unlock()
		}
	}()

	return &notification, nil
}

func (a *ConsoleApp) iconPath(kind NotificationKind) string {
	if a.iconsPath == "" {
		return ""
	}

	ext := ".png"
	if runtime.GOOS == "windows" {
		ext = ".ico"
	}

	switch kind {
	case NotificationKindAlert, NotificationKindUserInteraction:
		return filepath.Join(a.iconsPath, "skotty"+ext)
	case NotificationKindWarning:
		return filepath.Join(a.iconsPath, "skotty-warn"+ext)
	default:
		return ""
	}
}

func (a *ConsoleApp) urgency(kind NotificationKind) osNotifyOption {
	if kind == NotificationKindUserInteraction {
		return withOSNotifyUrgency(osNotificationUrgencyCritical)
	}

	return withOSNotifyUrgency(osNotificationUrgencyNormal)
}
