package ui

import (
	"sync"
	"unsafe"

	"golang.org/x/sys/windows"

	"a.yandex-team.ru/security/skotty/skotty/internal/ui/winapi"
)

var _ osNotifier = (*winGUINotifier)(nil)

type winGUINotifier struct {
	hWnd  windows.Handle
	guid  windows.GUID
	curID int
	mu    sync.Mutex
}

type winGUINotification struct {
	n  *winGUINotifier
	id int
}

func newGUINotifier(hWnd windows.Handle, guid windows.GUID) *winGUINotifier {
	return &winGUINotifier{
		hWnd: hWnd,
		guid: guid,
	}
}

func (n *winGUINotifier) Notify(title, text, iconPath string, _ ...osNotifyOption) (osNotification, error) {
	n.mu.Lock()
	defer n.mu.Unlock()

	data := n.newNotifyIconData()
	data.Flags |= winapi.NIF_INFO

	if iconPath != "" {
		icon, err := winapi.LoadImage(
			0, windows.StringToUTF16Ptr(iconPath),
			winapi.IMAGE_ICON,
			0, 0,
			winapi.LR_DEFAULTSIZE|winapi.LR_LOAD_FROM_FILE|winapi.LR_LOAD_TRANSPARENT|winapi.LR_SHARED,
		)

		if err == nil {
			data.Flags |= winapi.NIF_ICON
			data.Icon = icon
		}
	}

	if title != "" {
		copy(data.InfoTitle[:], windows.StringToUTF16(title))
	}

	copy(data.Info[:], windows.StringToUTF16(text))

	err := winapi.Shell_NotifyIcon(winapi.NIM_MODIFY, data)
	if err != nil {
		return nil, err
	}

	n.curID++
	return &winGUINotification{
		n:  n,
		id: n.curID,
	}, nil
}

func (n *winGUINotifier) Close() error {
	n.mu.Lock()
	defer n.mu.Unlock()

	data := n.newNotifyIconData()
	data.Flags |= winapi.NIIF_NONE
	return winapi.Shell_NotifyIcon(winapi.NIM_MODIFY, data)
}

func (n *winGUINotifier) newNotifyIconData() *winapi.NotifyIconData {
	var data winapi.NotifyIconData
	data.Size = uint32(unsafe.Sizeof(data))
	data.Version = winapi.NOTIFY_VERSION_4
	data.Flags = winapi.NIF_GUID
	data.Wnd = n.hWnd
	data.GuidItem = n.guid
	return &data
}

func (n *winGUINotifier) closeNotification(id int) error {
	n.mu.Lock()
	defer n.mu.Unlock()

	if n.curID != id {
		return nil
	}

	n.curID = 0
	data := n.newNotifyIconData()
	data.Flags |= winapi.NIIF_NONE
	return winapi.Shell_NotifyIcon(winapi.NIM_MODIFY, data)
}

func (n *winGUINotification) Close() {
	_ = n.n.closeNotification(n.id)
}
