package ui

import (
	"context"
	"fmt"
	"runtime"
	"unsafe"

	"golang.org/x/sys/windows"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/skotty/skotty/internal/ui/winapi"
	"a.yandex-team.ru/security/skotty/skotty/internal/version"
)

var _ Notifier = (*GraphicalApp)(nil)
var _ App = (*GraphicalApp)(nil)

type GraphicalApp struct {
	*ConsoleApp
	hInstance windows.Handle
	hWnd      windows.Handle
	hMenu     windows.Handle
	hTrayGUID windows.GUID
	windowCtx context.Context
	quitCmdID uint32
}

func NewGraphicalApp(opts ...Option) (*GraphicalApp, error) {
	ctx, cancel := context.WithCancel(context.Background())
	out := &GraphicalApp{
		ConsoleApp: &ConsoleApp{
			log:      &nop.Logger{},
			ctx:      ctx,
			cancelFn: cancel,
		},
	}
	var sockets []string
	for _, opt := range opts {
		switch v := opt.(type) {
		case optionLogger:
			out.log = v.log
		case optionIconsPath:
			out.iconsPath = v.iconsPath
		case optionSocketsNames:
			sockets = v.sockets
		}
	}

	if err := out.initializeWindow(); err != nil {
		cancel()
		return nil, fmt.Errorf("create window: %w", err)
	}

	if err := out.initializeTray(sockets...); err != nil {
		cancel()
		return nil, fmt.Errorf("create tray menu: %w", err)
	}

	out.notifier = newGUINotifier(out.hWnd, out.hTrayGUID)
	return out, nil
}

func (a *GraphicalApp) initializeWindow() error {

	windowCtx, windowCancel := context.WithCancel(context.Background())
	a.windowCtx = windowCtx

	// ugly hack to achieve: The window must belong to the current thread :(
	// https://docs.microsoft.com/en-us/windows/win32/api/winuser/nf-winuser-getmessagew
	errCh := make(chan error)
	go func() {
		runtime.LockOSThread()
		defer runtime.UnlockOSThread()
		defer windowCancel()

		createWindow := func() error {
			var err error
			a.hInstance, err = winapi.GetModuleHandle(nil)
			if err != nil {
				return fmt.Errorf("GetModuleHandlee: %w", err)
			}

			wndClass := windows.StringToUTF16Ptr("SkottyUIClass")
			wcex := winapi.WndClassEx{
				Style:      winapi.CS_HREDRAW | winapi.CS_VREDRAW,
				Instance:   a.hInstance,
				ClassName:  wndClass,
				Background: windows.Handle(winapi.COLOR_WINDOWFRAME),
				WndProc:    windows.NewCallback(a.wndProc),
			}
			wcex.Size = uint32(unsafe.Sizeof(wcex))
			if _, err := winapi.RegisterClassEx(&wcex); err != nil {
				return fmt.Errorf("RegisterClassEx: %w", err)
			}

			a.hWnd, err = winapi.CreateWindowEx(
				0,
				wndClass,
				windows.StringToUTF16Ptr("Skotty"),
				winapi.WS_OVERLAPPEDWINDOW,
				winapi.CW_USE_DEFAULT, winapi.CW_USE_DEFAULT,
				winapi.CW_USE_DEFAULT, winapi.CW_USE_DEFAULT,
				0,
				0,
				a.hInstance,
				nil,
			)
			if err != nil {
				return fmt.Errorf("CreateWindowEx: %w", err)
			}

			_ = winapi.UpdateWindow(a.hWnd)
			return nil
		}

		if err := createWindow(); err != nil {
			errCh <- err
			close(errCh)
			return
		}

		close(errCh)
		var msg winapi.Msg
		for {
			rv, _ := winapi.GetMessage(&msg, a.hWnd, 0, 0)
			if rv <= 0 {
				break
			}

			_ = winapi.TranslateMessage(&msg)
			winapi.DispatchMessage(&msg)
		}
	}()

	return <-errCh
}

func (a *GraphicalApp) initializeTray(sockets ...string) error {
	a.hTrayGUID = newGUID()
	nid := &winapi.NotifyIconData{
		Wnd:             a.hWnd,
		ID:              100,
		GuidItem:        a.hTrayGUID,
		Flags:           winapi.NIF_MESSAGE | winapi.NIF_GUID | winapi.NIF_TIP,
		CallbackMessage: winapi.WM_SYSTRAY,
		State:           winapi.NIS_SHAREDICON,
		StateMask:       winapi.NIS_SHAREDICON,
	}
	nid.Size = uint32(unsafe.Sizeof(*nid))
	copy(nid.Tip[:], windows.StringToUTF16(AppName))

	icon, err := winapi.LoadImage(
		0, windows.StringToUTF16Ptr(a.iconPath(NotificationKindAlert)),
		winapi.IMAGE_ICON,
		0, 0,
		winapi.LR_DEFAULTSIZE|winapi.LR_LOAD_FROM_FILE|winapi.LR_LOAD_TRANSPARENT|winapi.LR_SHARED,
	)
	if err == nil {
		nid.Flags |= winapi.NIF_ICON
		nid.Icon = icon
	}

	err = winapi.Shell_NotifyIcon(winapi.NIM_ADD, nid)
	if err != nil {
		return fmt.Errorf("failed to add tray icon: %w", err)
	}

	a.hMenu, err = winapi.CreatePopupMenu()
	if err != nil {
		return fmt.Errorf("failed to create popup menu: %w", err)
	}

	items := []menuItem{
		{
			title:    "enabled sockets",
			disabled: true,
		},
		{
			separator: true,
		},
	}

	for _, item := range sockets {
		items = append(items, menuItem{
			title:    item,
			disabled: true,
		})
	}

	items = append(items,
		menuItem{
			separator: true,
		},
		menuItem{
			title:    fmt.Sprintf("version: %s", version.Full()),
			disabled: true,
		},
		menuItem{
			title: "quit",
		},
	)

	mb := &menuBuilder{
		handle: a.hMenu,
	}

	for _, item := range items {
		a.quitCmdID, err = mb.addMenuItem(item)
		if err != nil {
			return err
		}
	}

	return nil
}

func (a *GraphicalApp) wndProc(hwnd windows.Handle, msg uint32, wParam, lParam uintptr) uintptr {
	process := func() (bool, error) {
		switch msg {
		case winapi.WM_CLOSE:
			_ = winapi.DestroyWindow(a.hWnd)
		case winapi.WM_COMMAND:
			menuItemId := int32(wParam)
			// https://docs.microsoft.com/en-us/windows/win32/menurc/wm-command#menus
			if uint32(menuItemId) == a.quitCmdID {
				_ = winapi.PostMessage(a.hWnd, winapi.WM_CLOSE, 0, 0)
			}
		case winapi.WM_SYSTRAY:
			if winapi.LoWord(uint32(lParam)) != winapi.WM_RBUTTONUP {
				return true, nil
			}

			var pt winapi.Point
			if err := winapi.GetCursorPos(&pt); err != nil {
				return false, fmt.Errorf("GetCursorPos: %w", err)
			}

			if err := winapi.SetForegroundWindow(a.hWnd); err != nil {
				return false, fmt.Errorf("SetForegroundWindow: %w", err)
			}

			_, err := winapi.TrackPopupMenu(
				a.hMenu,
				winapi.TPM_RIGHTBUTTON|winapi.TPM_RIGHTALIGN|winapi.TPM_NONOTIFY,
				pt.X, pt.Y,
				0,
				a.hWnd,
				0,
			)
			if err != nil {
				return false, fmt.Errorf("TrackPopupMenu: %w", err)
			}

			_ = winapi.PostMessage(a.hWnd, winapi.WM_NULL, 0, 0)
			return false, nil
		}

		return true, nil
	}

	fwd, err := process()
	if err != nil {
		a.log.Error("can't process window proc", log.Error(err))
	}

	if fwd {
		return winapi.DefWindowProc(hwnd, msg, wParam, lParam)
	}

	return 0
}

func (a *GraphicalApp) Start(mainFn MainFn) error {
	return a.ConsoleApp.start(a.windowCtx, mainFn)
}

func (a *GraphicalApp) Shutdown(ctx context.Context) error {
	if a.hWnd != 0 {
		nid := winapi.NotifyIconData{
			Flags:    winapi.NIF_GUID,
			Wnd:      a.hWnd,
			GuidItem: a.hTrayGUID,
		}
		nid.Size = uint32(unsafe.Sizeof(nid))

		_ = winapi.DestroyMenu(a.hMenu)
		_ = winapi.Shell_NotifyIcon(winapi.NIM_DELETE, &nid)
		_ = winapi.PostMessage(a.hWnd, winapi.WM_CLOSE, 0, 0)
	}

	return a.ConsoleApp.Shutdown(ctx)
}
