//go:build linux || freebsd
// +build linux freebsd

package ui

import (
	"context"
	"fmt"
	"time"

	"github.com/godbus/dbus/v5"
)

const (
	notifyObjectPath            = "/org/freedesktop/Notifications"
	notifyInterfacePath         = "org.freedesktop.Notifications"
	notifyCallCloseNotification = "org.freedesktop.Notifications.CloseNotification"
	notifyCallNotify            = "org.freedesktop.Notifications.Notify"

	hintUrgency          = "urgency"
	urgencyLow      byte = 0
	urgencyNormal   byte = 1
	urgencyCritical byte = 2
)

var (
	defReplacesID uint32
	defActions    []string
	defTimeout    = int32(-1)
	callTimeout   = 1 * time.Second
)

var _ osNotifier = (*linuxNotifier)(nil)

type linuxNotifier struct {
	dbusConn *dbus.Conn
}

type linuxNotification struct {
	id       uint32
	dbusConn *dbus.Conn
}

func newNotifier() (osNotifier, error) {
	conn, err := dbus.ConnectSessionBus()
	if err != nil {
		return nil, fmt.Errorf("failed to connect dbus: %w", err)
	}

	return &linuxNotifier{
		dbusConn: conn,
	}, nil
}

func (n *linuxNotifier) Notify(title, text, iconPath string, opts ...osNotifyOption) (osNotification, error) {
	hints := map[string]dbus.Variant{}
	for _, opt := range opts {
		switch o := opt.(type) {
		case osNotifierOptionUrgency:
			hints[hintUrgency] = urgencyToHint(o.urgency)
		}
	}

	obj := n.dbusConn.Object(notifyInterfacePath, notifyObjectPath)
	rsp := dbusCall(
		obj,
		notifyCallNotify,
		AppName,
		defReplacesID,
		iconPath,
		title,
		text,
		defActions,
		hints,
		defTimeout,
	)

	if rsp.Err != nil {
		return nil, fmt.Errorf("can't call dbus notify: %w", rsp.Err)
	}

	var id uint32
	_ = rsp.Store(&id)
	return &linuxNotification{
		id:       id,
		dbusConn: n.dbusConn,
	}, nil
}

func (n *linuxNotifier) Close() error {
	return n.dbusConn.Close()
}

func (n *linuxNotification) Close() {
	if n.id == 0 {
		return
	}

	obj := n.dbusConn.Object(notifyInterfacePath, notifyObjectPath)
	_ = dbusCall(obj, notifyCallCloseNotification, n.id)
}

func dbusCall(obj dbus.BusObject, method string, args ...interface{}) *dbus.Call {
	ctx, cancel := context.WithTimeout(context.Background(), callTimeout)
	defer cancel()
	return obj.CallWithContext(ctx, method, 0, args...)
}

func urgencyToHint(u osNotifyUrgency) dbus.Variant {
	switch u {
	case osNotificationUrgencyCritical:
		return dbus.MakeVariant(urgencyCritical)
	default:
		return dbus.MakeVariant(urgencyNormal)
	}
}
