package ui

import (
	"crypto/rand"
	"fmt"
	"sync"
	"unsafe"

	"golang.org/x/sys/windows"

	"a.yandex-team.ru/security/skotty/skotty/internal/ui/winapi"
)

var _ osNotifier = (*winNotifier)(nil)

type winNotifier struct {
	hWnd  windows.Handle
	guid  windows.GUID
	curID int
	mu    sync.Mutex
}

type winNotification struct {
	n  *winNotifier
	id int
}

func newNotifierWithWindow(hWnd windows.Handle) (osNotifier, error) {
	notifier := &winNotifier{
		hWnd: hWnd,
		guid: newGUID(),
	}

	data := notifier.newNotifyIconData()
	data.Flags |= winapi.NIF_STATE
	data.State = winapi.NIS_HIDDEN
	data.StateMask = winapi.NIS_HIDDEN | winapi.NIS_SHAREDICON
	if err := winapi.Shell_NotifyIcon(winapi.NIM_ADD, data); err != nil {
		return nil, err
	}
	return notifier, nil
}

func newNotifier() (osNotifier, error) {
	hInstance, err := winapi.GetModuleHandle(nil)
	if err != nil {
		return nil, fmt.Errorf("failed to get module handle: %w", err)
	}

	wndClass := windows.StringToUTF16Ptr("SkottyNotifyClass")
	wcex := winapi.WndClassEx{
		Instance:  hInstance,
		ClassName: wndClass,
		WndProc: windows.NewCallback(func(hwnd windows.Handle, msg uint32, wParam, lParam uintptr) uintptr {
			switch msg {
			case winapi.WM_DESTROY:
				winapi.PostQuitMessage(0)
			default:
				return winapi.DefWindowProc(hwnd, msg, wParam, lParam)
			}
			return 0
		}),
	}
	wcex.Size = uint32(unsafe.Sizeof(wcex))
	if _, err := winapi.RegisterClassEx(&wcex); err != nil {
		return nil, fmt.Errorf("failed to register window class: %w", err)
	}

	hWnd, err := winapi.CreateWindowEx(
		0,
		wndClass,
		windows.StringToUTF16Ptr("SkottyNotifier"),
		winapi.WS_OVERLAPPED|winapi.WS_SYSMENU,
		winapi.CW_USE_DEFAULT, winapi.CW_USE_DEFAULT,
		winapi.CW_USE_DEFAULT, winapi.CW_USE_DEFAULT,
		0,
		0,
		hInstance,
		nil,
	)
	if err != nil {
		return nil, fmt.Errorf("failed to create window: %w", err)
	}

	return newNotifierWithWindow(hWnd)
}

func (n *winNotifier) Notify(title, text, iconPath string, _ ...osNotifyOption) (osNotification, error) {
	n.mu.Lock()
	defer n.mu.Unlock()

	data := n.newNotifyIconData()
	data.Flags |= winapi.NIF_INFO | winapi.NIF_STATE
	data.StateMask = winapi.NIS_HIDDEN

	if iconPath != "" {
		icon, err := winapi.LoadImage(
			0, windows.StringToUTF16Ptr(iconPath),
			winapi.IMAGE_ICON,
			0, 0,
			winapi.LR_DEFAULTSIZE|winapi.LR_LOAD_FROM_FILE|winapi.LR_LOAD_TRANSPARENT|winapi.LR_SHARED,
		)

		if err == nil {
			data.Flags |= winapi.NIF_ICON
			data.Icon = icon
		}
	}

	if title != "" {
		copy(data.InfoTitle[:], windows.StringToUTF16(title))
	}

	copy(data.Info[:], windows.StringToUTF16(text))

	err := winapi.Shell_NotifyIcon(winapi.NIM_MODIFY, data)
	if err != nil {
		return nil, err
	}

	n.curID++
	return &winNotification{
		n:  n,
		id: n.curID,
	}, nil
}

func (n *winNotifier) Close() error {
	n.mu.Lock()
	defer n.mu.Unlock()

	return winapi.Shell_NotifyIcon(winapi.NIM_DELETE, n.newNotifyIconData())
}

func (n *winNotifier) closeNotification(id int) error {
	n.mu.Lock()
	defer n.mu.Unlock()

	if n.curID != id {
		return nil
	}

	n.curID = 0
	data := n.newNotifyIconData()
	data.Flags |= winapi.NIF_STATE
	data.State = winapi.NIS_HIDDEN
	data.StateMask = winapi.NIS_HIDDEN
	return winapi.Shell_NotifyIcon(winapi.NIM_MODIFY, data)
}

func (n *winNotifier) newNotifyIconData() *winapi.NotifyIconData {
	var data winapi.NotifyIconData
	data.Size = uint32(unsafe.Sizeof(data))
	data.Version = winapi.NOTIFY_VERSION_4
	data.Flags = winapi.NIF_GUID
	data.Wnd = n.hWnd
	data.GuidItem = n.guid
	return &data
}

func (n *winNotification) Close() {
	_ = n.n.closeNotification(n.id)
}

func newGUID() windows.GUID {
	var buf [16]byte
	_, _ = rand.Read(buf[:])
	return *(*windows.GUID)(unsafe.Pointer(&buf[0]))
}
