package yubiconf

import (
	"fmt"

	"a.yandex-team.ru/security/libs/go/pcsc"
)

var (
	aidManagement = [...]byte{0xa0, 0x00, 0x00, 0x05, 0x27, 0x47, 0x11, 0x17}
	aidPIV        = [...]byte{0xa0, 0x00, 0x00, 0x03, 0x08}
)

const (
	insSelectApplication = 0xa4
	insWriteConfig       = 0x1c
	insReadConfig        = 0x1d
	insGetData           = 0xcb
	insPutData           = 0xdb
	insGetVersion        = 0xfd
	insVerify            = 0x20
)

type PCSCClient struct {
	scClient pcsc.Client
	scHandle pcsc.Handle
	scTx     pcsc.Tx
}

func NewPCSCClient(card Card) (*PCSCClient, error) {
	pc, err := pcsc.NewClient()
	if err != nil {
		return nil, fmt.Errorf("connecting to smart card daemon: %w", err)
	}

	h, err := pc.Connect(card.Name)
	if err != nil {
		_ = pc.Close()
		return nil, fmt.Errorf("connecting to smart card: %w", err)
	}

	tx, err := h.Begin()
	if err != nil {
		return nil, fmt.Errorf("beginning smart card transaction: %w", err)
	}

	prepare := func() error {
		if err := ykSelectApplication(tx, aidPIV[:]); err != nil {
			return fmt.Errorf("selecting piv applet: %w", err)
		}

		if err := ykCheckVersion(tx); err != nil {
			return fmt.Errorf("check version: %w", err)
		}

		return nil
	}

	if err := prepare(); err != nil {
		_ = tx.Close()
		_ = h.Close()
		_ = pc.Close()
		return nil, err
	}

	return &PCSCClient{
		scClient: pc,
		scHandle: h,
		scTx:     tx,
	}, nil
}

func (p *PCSCClient) Login(pin string) error {
	data, err := encodePIN(pin)
	if err != nil {
		return err
	}

	cmd := pcsc.APDU{
		Instruction: insVerify,
		Param2:      0x80,
		Data:        data,
	}
	if _, err := p.scTx.Transmit(cmd); err != nil {
		return fmt.Errorf("verify pin: %w", err)
	}
	return nil
}

func (p *PCSCClient) ReadObject(object uint32) ([]byte, error) {
	cmd := pcsc.APDU{
		Instruction: insGetData,
		Param1:      0x3f,
		Param2:      0xff,
		Data: []byte{
			0x5c, // Tag list
			0x03, // Length of tag
			byte(object >> 16),
			byte(object >> 8),
			byte(object),
		},
	}
	return p.scTx.Transmit(cmd)
}

func (p *PCSCClient) Close() {
	_ = p.scTx.Close()
	_ = p.scHandle.Close()
	_ = p.scClient.Close()
}

func ykSelectApplication(tx pcsc.Tx, id []byte) error {
	cmd := pcsc.APDU{
		Instruction: insSelectApplication,
		Param1:      0x04,
		Data:        id[:],
	}
	if _, err := tx.Transmit(cmd); err != nil {
		return fmt.Errorf("command failed: %w", err)
	}
	return nil
}

func ykCheckVersion(tx pcsc.Tx) error {
	cmd := pcsc.APDU{
		Instruction: insGetVersion,
	}
	resp, err := tx.Transmit(cmd)
	if err != nil {
		return fmt.Errorf("can't check Yubikey version: command failed: %w", err)
	}

	if n := len(resp); n != 3 {
		return fmt.Errorf("can't check Yubikey version: expected response to have 3 bytes, got: %d", n)
	}

	if resp[0] != 5 {
		return fmt.Errorf("supported only Yubikey 5 version (see https://st.yandex-team.ru/SKOTTY-34), current version: %d.%d.%d", resp[0], resp[1], resp[2])
	}
	return nil
}

func encodePIN(pin string) ([]byte, error) {
	data := []byte(pin)
	if len(data) == 0 {
		return nil, fmt.Errorf("pin cannot be empty")
	}
	if len(data) > 8 {
		return nil, fmt.Errorf("pin longer than 8 bytes")
	}
	// apply padding
	for i := len(data); i < 8; i++ {
		data = append(data, 0xff)
	}
	return data, nil
}
