package yubiconf

import (
	"bytes"
	"encoding/asn1"
	"encoding/binary"
	"fmt"
	"io"

	"a.yandex-team.ru/security/libs/go/pcsc"
)

type YubiConf struct {
	pc      *PCSCClient
	modTags map[ConfigTag]struct{}
	cfg     map[ConfigTag][]byte
}

func Open(card Card) (*YubiConf, error) {
	pc, err := NewPCSCClient(card)
	if err != nil {
		return nil, err
	}

	if err := ykSelectApplication(pc.scTx, aidManagement[:]); err != nil {
		_ = pc.Close
		return nil, fmt.Errorf("selecting management applet: %w", err)
	}

	y := &YubiConf{
		pc:      pc,
		modTags: make(map[ConfigTag]struct{}),
		cfg:     make(map[ConfigTag][]byte),
	}

	if err := y.readConfig(); err != nil {
		y.Close()
		return nil, fmt.Errorf("read config: %w", err)
	}

	return y, nil
}

func (y *YubiConf) Close() {
	y.pc.Close()
}

func (y *YubiConf) IsUSBAppSupported(app ApplicationID) bool {
	supported := binary.BigEndian.Uint16(y.cfg[ConfigTagUsbSupported])
	return uint16(app)&supported != 0
}

func (y *YubiConf) IsUSBAppEnabled(app ApplicationID) bool {
	if !y.IsUSBAppSupported(app) {
		return false
	}

	enabled := binary.BigEndian.Uint16(y.cfg[ConfigTagUsbEnabled])
	return uint16(app)&enabled != 0
}

func (y *YubiConf) SupportedUSBApps() []ApplicationID {
	supported := binary.BigEndian.Uint16(y.cfg[ConfigTagUsbSupported])
	var out []ApplicationID
	for _, app := range AllApplications {
		if uint16(app)&supported == 0 {
			continue
		}

		out = append(out, app)
	}

	return out
}

func (y *YubiConf) EnabledUSBApps() []ApplicationID {
	enabled := binary.BigEndian.Uint16(y.cfg[ConfigTagUsbEnabled])
	var out []ApplicationID
	for _, app := range AllApplications {
		if uint16(app)&enabled == 0 {
			continue
		}

		out = append(out, app)
	}

	return out
}

func (y *YubiConf) EnableUSBApp(app ApplicationID) error {
	if !y.IsUSBAppSupported(app) {
		return fmt.Errorf("USB application %s in not supported by yubikey", app)
	}

	enabled := binary.BigEndian.Uint16(y.cfg[ConfigTagUsbEnabled])
	enabled |= uint16(app)
	binary.BigEndian.PutUint16(y.cfg[ConfigTagUsbEnabled], enabled)
	y.modTags[ConfigTagUsbEnabled] = struct{}{}
	return nil
}

func (y *YubiConf) DisableUSBApp(app ApplicationID) error {
	if !y.IsUSBAppSupported(app) {
		return fmt.Errorf("USB application %s in not supported by yubikey", app)
	}

	enabled := binary.BigEndian.Uint16(y.cfg[ConfigTagUsbEnabled])
	enabled &= ^uint16(app)
	binary.BigEndian.PutUint16(y.cfg[ConfigTagUsbEnabled], enabled)
	y.modTags[ConfigTagUsbEnabled] = struct{}{}
	return nil
}

func (y *YubiConf) WriteConfig(reboot bool) error {
	if len(y.modTags) == 0 {
		// nothing to write
		return nil
	}

	var config bytes.Buffer
	if reboot {
		valBytes, err := asn1.Marshal(asn1.RawValue{
			Tag: int(ConfigTagReboot),
		})
		if err != nil {
			return fmt.Errorf("failed to marshal reboot tag: %w", err)
		}
		config.Write(valBytes)
	}

	for tag := range y.modTags {
		valBytes, err := asn1.Marshal(asn1.RawValue{
			Tag:   int(tag),
			Bytes: y.cfg[tag],
		})
		if err != nil {
			return fmt.Errorf("failed to marshal cfg tag 0x%x: %w", tag, err)
		}
		config.Write(valBytes)
	}

	return y.writeConfig(config.Bytes())
}

func (y *YubiConf) Reboot() error {
	config, err := asn1.Marshal(asn1.RawValue{
		Tag: int(ConfigTagReboot),
	})
	if err != nil {
		return fmt.Errorf("failed to marshal reboot tag: %w", err)
	}

	return y.writeConfig(config)
}

func (y *YubiConf) writeConfig(config []byte) error {
	if len(config) > 254 {
		return fmt.Errorf("config is too large (must be up to 254 bytes): %d", len(config))
	}

	req := bytes.NewBuffer([]byte{
		byte(len(config)),
	})
	req.Write(config)

	cmd := pcsc.APDU{
		Instruction: insWriteConfig,
		Data:        req.Bytes(),
	}

	_, err := y.pc.scTx.Transmit(cmd)
	return err
}

func (y *YubiConf) readConfig() error {
	cmd := pcsc.APDU{
		Instruction: insReadConfig,
	}

	resp, err := y.pc.scTx.Transmit(cmd)
	if err != nil {
		return err
	}

	if len(resp) < 2 {
		return fmt.Errorf("too short read config response: %d", len(resp))
	}

	var respData bytes.Buffer
	if _, err := io.CopyN(&respData, bytes.NewReader(resp[1:]), int64(resp[0])); err != nil {
		return fmt.Errorf("failed to read read config response: %w", err)
	}

	data := respData.Bytes()
	var cfg asn1.RawValue
	for len(data) > 0 {
		data, err = asn1.Unmarshal(data, &cfg)
		if err != nil {
			return fmt.Errorf("failed to unmarshal config: %w", err)
		}

		cfgTag := ConfigTag(cfg.Tag)
		if !isConfigTagSupported(cfgTag) {
			continue
		}

		y.cfg[cfgTag] = cfg.Bytes
	}

	return nil
}
