package yubikey

import (
	"crypto/x509"
	"fmt"

	"a.yandex-team.ru/security/libs/go/piv"
)

type Slot struct {
	PIVSlot piv.Slot
}

type AuthErr = piv.AuthErr

var (
	ErrNoManagementKey   = piv.ErrNoManagementKey
	DefaultPIN           = piv.DefaultPIN
	DefaultPUK           = piv.DefaultPUK
	DefaultManagementKey = piv.DefaultManagementKey

	IsAuthErr = piv.IsAuthErr

	AllSlots = func() []Slot {
		out := []Slot{
			MustSlotFromKeyID(piv.SlotAuthentication.Key),
			MustSlotFromKeyID(piv.SlotSignature.Key),
			MustSlotFromKeyID(piv.SlotCardAuthentication.Key),
		}

		for id := uint32(0x82); id <= 0x95; id++ {
			out = append(out, MustSlotFromKeyID(id))
		}

		return out
	}()
)

type Cert struct {
	*x509.Certificate
	Slot Slot
}

func (s Slot) String() string {
	return fmt.Sprintf("0x%02x", s.PIVSlot.Key)
}

func MustSlotFromKeyID(keyID uint32) Slot {
	out, err := SlotFromKeyID(keyID)
	if err != nil {
		panic(err.Error())
	}
	return out
}

func SlotFromKeyID(keyID uint32) (Slot, error) {
	switch keyID {
	case piv.SlotAuthentication.Key:
		return Slot{PIVSlot: piv.SlotAuthentication}, nil
	case piv.SlotSignature.Key:
		return Slot{PIVSlot: piv.SlotSignature}, nil
	case piv.SlotCardAuthentication.Key:
		return Slot{PIVSlot: piv.SlotCardAuthentication}, nil
	}

	pivSlot, ok := piv.RetiredKeyManagementSlot(keyID)
	if !ok {
		return Slot{}, fmt.Errorf("unsupported key id: 0x%02x", keyID)
	}

	return Slot{PIVSlot: pivSlot}, nil
}
