package yubikey

import (
	"crypto"
	"crypto/rand"
	"crypto/x509"
	"errors"
	"fmt"
	"io"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/libs/go/piv"
	"a.yandex-team.ru/security/skotty/skotty/internal/certgen"
	"a.yandex-team.ru/security/skotty/skotty/internal/logger"
)

type Yubikey struct {
	Card
	PIN          string
	yk           *piv.YubiKey
	haveMngmtKey bool
	mngmtKey     *piv.ManagementKey
}

type CertRequest struct {
	piv.Key
	CommonName string
}

func OpenBySerial(serial uint32) (*Yubikey, error) {
	cards, err := Cards()
	if err != nil {
		return nil, fmt.Errorf("can't listen yubikeys: %w", err)
	}

	var card Card
	for _, candidate := range cards {
		if candidate.Serial == serial {
			card = candidate
			break
		}
	}

	if card.Serial == 0 {
		return nil, fmt.Errorf("yubikey with serial %d is not found", serial)
	}

	return Open(card)
}

func Open(card Card) (*Yubikey, error) {
	yk, err := piv.Open(card.Name)
	if err != nil {
		return nil, err
	}

	ykVer := yk.Version()
	if ykVer.Major != 5 {
		_ = yk.Close()
		return nil, fmt.Errorf("supported only Yubikey 5 version (see https://st.yandex-team.ru/SKOTTY-34), current version: %s", ykVer)
	}

	if card.Serial != 0 {
		serial, err := yk.Serial()
		if err != nil {
			_ = yk.Close()
			return nil, fmt.Errorf("failed to get serial number: %w", err)
		}

		if card.Serial != serial {
			_ = yk.Close()
			return nil, fmt.Errorf("serial number mismatch: %d != %d", card.Serial, serial)
		}
	}

	return &Yubikey{
		Card: card,
		yk:   yk,
	}, nil
}

func (y *Yubikey) SetPIN(old, new string) error {
	return y.yk.SetPIN(old, new)
}

func (y *Yubikey) SetPUK(old, new string) error {
	return y.yk.SetPUK(old, new)
}

func (y *Yubikey) Reset(newPIN, newPUK string) error {
	err := y.yk.Reset()
	if err != nil {
		return fmt.Errorf("reset failed: %w", err)
	}

	err = y.ResetMngmtKey(&DefaultManagementKey)
	if err != nil {
		return fmt.Errorf("unable to reset management key: %w", err)
	}

	if err := y.yk.SetPUK(piv.DefaultPUK, newPUK); err != nil {
		return fmt.Errorf("failed to change PUK key: %w", err)
	}

	if err := y.yk.SetPIN(piv.DefaultPIN, newPIN); err != nil {
		return fmt.Errorf("failed to change PIN: %w", err)
	}

	return nil
}

func (y *Yubikey) Unblock(puk, newPIN string) error {
	return y.yk.Unblock(puk, newPIN)
}

func (y *Yubikey) GenCertificate(slot Slot, pin string, req CertRequest) (*x509.Certificate, error) {
	mngmtKey, err := y.MngmtKey(pin)
	if err != nil {
		return nil, err
	}

	pub, err := y.yk.GenerateKey(mngmtKey, slot.PIVSlot, req.Key)
	if err != nil {
		return nil, err
	}

	return y.setupCertificate(slot, pin, req.CommonName, pub)
}

func (y *Yubikey) setupCertificate(slot Slot, pin string, commonName string, pub crypto.PublicKey) (*x509.Certificate, error) {
	certBytes, err := certgen.GenCertificateFor(commonName, pub)
	if err != nil {
		return nil, err
	}

	cert, err := x509.ParseCertificate(certBytes)
	if err != nil {
		return nil, fmt.Errorf("couldn't parse certificate: %w", err)
	}

	if err := y.SetCertificate(slot, pin, cert); err != nil {
		return nil, fmt.Errorf("couldn't update certificate: %w", err)
	}

	return cert, nil
}

func (y *Yubikey) ListKeys(slots ...Slot) ([]Cert, error) {
	if len(slots) == 0 {
		slots = AllSlots
	}

	var out []Cert
	for _, slot := range slots {
		cert, err := y.Certificate(slot)
		if err != nil {
			if !errors.Is(err, piv.ErrNotFound) {
				logger.Error("failed to get certificate",
					log.String("card", y.Card.String()),
					log.String("slot", slot.String()),
					log.Error(err),
				)
			}

			continue
		}

		out = append(out, Cert{
			Certificate: cert,
			Slot:        slot,
		})
	}

	return out, nil
}

func (y *Yubikey) ResetMngmtKey(key *piv.ManagementKey) error {
	newKey := piv.ManagementKey{
		ManagementKeyMetadata: piv.ManagementKeyMetadata{
			Algo:        piv.ManagementKeyAlgoTDES,
			TouchPolicy: piv.TouchPolicyNever,
		},
		Key: make([]byte, piv.ManagementKeyAlgoTDES.KeyLen()),
	}

	if _, err := io.ReadFull(rand.Reader, newKey.Key); err != nil {
		return fmt.Errorf("failed to generate new management key: %w", err)
	}

	if err := y.yk.SetManagementKey(key, &newKey); err != nil {
		return fmt.Errorf("failed to change management key: %w", err)
	}

	// Store management key on the YubiKey.
	pm := &piv.ProtectedMetadata{
		ManagementKey: newKey.Key,
	}
	if err := y.yk.SetProtectedMetadata(&newKey, pm); err != nil {
		return fmt.Errorf("failed to update protected metadata: %w", err)
	}

	// Store metadata to be compatible with other apps (yc for e.g.)
	m := &piv.Metadata{
		Flags: piv.MetadataFlagMgmKeyProtected,
	}
	if err := y.yk.SetMetadata(m); err != nil {
		return fmt.Errorf("failed to update metadata: %w", err)
	}

	return nil
}

func (y *Yubikey) MngmtKey(pin string) (*piv.ManagementKey, error) {
	if y.haveMngmtKey {
		return y.mngmtKey, nil
	}

	key, err := y.yk.ManagementKey(pin)
	if err != nil {
		return nil, err
	}

	y.haveMngmtKey = true
	y.mngmtKey = key
	return y.mngmtKey, nil
}

func (y *Yubikey) PrivateKey(slot Slot, keyAuth piv.KeyAuth, public crypto.PublicKey) (crypto.Signer, error) {
	return y.yk.PrivateKey(slot.PIVSlot, public, keyAuth)
}

func (y *Yubikey) Attest(slot Slot) (*x509.Certificate, error) {
	return y.yk.Attest(slot.PIVSlot)
}

func (y *Yubikey) Certificate(slot Slot) (*x509.Certificate, error) {
	return y.yk.Certificate(slot.PIVSlot)
}

func (y *Yubikey) SetCertificate(slot Slot, pin string, cert *x509.Certificate) error {
	mngmtKey, err := y.MngmtKey(pin)
	if err != nil {
		return err
	}

	return y.yk.SetCertificate(mngmtKey, slot.PIVSlot, cert)
}

func (y *Yubikey) AttestationCertificate() (*x509.Certificate, error) {
	return y.yk.AttestationCertificate()
}

func (y *Yubikey) Version() piv.Version {
	return y.yk.Version()
}

func (y *Yubikey) IsManagementKeyValid(key *piv.ManagementKey) bool {
	return y.yk.CheckManagementKey(key) == nil
}

func (y *Yubikey) Close() {
	_ = y.yk.Close()
}
