package osutil

import (
	"bytes"
	_ "embed"
	"errors"
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"strings"
	"text/template"

	"a.yandex-team.ru/security/skotty/skotty/internal/paths"
)

//go:embed template/launchd.service
var serviceTmpl string

const serviceName = "ru.yandex.skotty"

func CheckService() error {
	_, err := callLaunchCtl("list", serviceName)
	if err != nil {
		return ErrNotInstalled
	}
	return nil
}

func Restart() error {
	if err := CheckService(); err != nil {
		return err
	}

	// stop job_label
	//   Stop the specified job by label.  Non-demand based jobs will always be restarted.
	//   Use of this subcommand is discouraged.
	_, err := callLaunchCtl("stop", serviceName)
	if err != nil {
		return fmt.Errorf("can't restart launchd service: %w", err)
	}

	return nil
}

func InstallService() (string, error) {
	exePath, err := Executable()
	if err != nil {
		return "", fmt.Errorf("skotty executable not found: %w", err)
	}

	servicePath, err := paths.Service()
	if err != nil {
		return "", fmt.Errorf("can't determine launchd path: %w", err)
	}

	err = os.MkdirAll(filepath.Dir(servicePath), 0o700)
	if err != nil {
		return "", err
	}

	if _, err := os.Stat(servicePath); err == nil {
		_, _ = callLaunchCtl("unload", servicePath)
	}

	serviceFile, err := os.Create(servicePath)
	if err != nil {
		return "", fmt.Errorf("can't render launchd job file: %w", err)
	}

	serviceInfo := struct {
		Args []string
	}{
		Args: []string{
			exePath,
			"start",
		},
	}

	err = template.Must(template.New("service").Parse(serviceTmpl)).Execute(serviceFile, serviceInfo)
	if err != nil {
		return "", fmt.Errorf("failed to render launchd job file: %w", err)
	}

	err = serviceFile.Close()
	if err != nil {
		return "", err
	}

	_, err = callLaunchCtl("load", "-w", servicePath)
	if err != nil {
		return "", fmt.Errorf("can't enable launchd service: %w", err)
	}

	return "", nil
}

func UnInstallService() error {
	if err := CheckService(); err != nil {
		// not installed
		return nil
	}

	servicePath, err := paths.Service()
	if err != nil {
		return fmt.Errorf("can't determine launchd service path: %w", err)
	}

	_, err = callLaunchCtl("unload", servicePath)
	if err != nil {
		return err
	}

	err = os.RemoveAll(servicePath)
	if err != nil {
		if os.IsNotExist(err) {
			return nil
		}
		return err
	}

	return nil
}

func callLaunchCtl(args ...string) (string, error) {
	c := exec.Command("launchctl", args...)
	var stdout, stdin, stderr bytes.Buffer
	c.Stdin = &stdin
	c.Stdout = &stdout
	c.Stderr = &stderr

	if err := c.Run(); err != nil {
		_, _ = stderr.WriteTo(os.Stderr)
		return "", fmt.Errorf("failed to execute: launchctl %s: %w", c, err)
	}

	strErr := stderr.String()
	_, _ = os.Stderr.WriteString(strErr)
	// яблочники как всегда
	if strings.Contains(strErr, "Load failed") {
		return "", errors.New("load failed")
	}

	return stdout.String(), nil
}
