package osutil

import (
	"bytes"
	_ "embed"
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"text/template"

	"a.yandex-team.ru/security/skotty/skotty/internal/paths"
)

//go:embed template/systemd.service
var serviceTmpl string

func CheckService() error {
	_, err := callSystemCtl(true, "is-enabled", "skotty")
	if err != nil {
		return ErrNotInstalled
	}

	return nil
}

func Restart() error {
	if err := CheckService(); err != nil {
		return err
	}

	_, err := callSystemCtl(false, "restart", "skotty")
	if err != nil {
		return err
	}

	return nil
}

func InstallService() (string, error) {
	servicePath, err := paths.Service()
	if err != nil {
		return "", fmt.Errorf("can't determine systemd service path: %w", err)
	}

	exePath, err := Executable()
	if err != nil {
		return "", fmt.Errorf("skotty executable not found: %w", err)
	}

	err = os.MkdirAll(filepath.Dir(servicePath), 0o700)
	if err != nil {
		return "", err
	}

	serviceFile, err := os.Create(servicePath)
	if err != nil {
		return "", fmt.Errorf("can't create systemd service file: %w", err)
	}

	serviceInfo := struct {
		CMD string
	}{
		CMD: fmt.Sprintf("%s start", exePath),
	}

	err = template.Must(template.New("service").Parse(serviceTmpl)).Execute(serviceFile, serviceInfo)
	if err != nil {
		return "", fmt.Errorf("failed to render systemd service file: %w", err)
	}

	err = serviceFile.Close()
	if err != nil {
		return "", err
	}

	_, err = callSystemCtl(false, "enable", servicePath)
	if err != nil {
		return "", fmt.Errorf("can't enable systemd service: %w", err)
	}

	_, err = callSystemCtl(false, "restart", "skotty")
	if err != nil {
		return "", fmt.Errorf("can't start skotty systemd unit: %w", err)
	}

	return "", nil
}

func UnInstallService() error {
	_, err := callSystemCtl(false, "disable", "skotty")
	return err
}

func callSystemCtl(suppressStderr bool, args ...string) (string, error) {
	systemctlArgs := []string{
		"--user",
	}
	systemctlArgs = append(systemctlArgs, args...)

	c := exec.Command("systemctl", systemctlArgs...)
	var stdout, stdin bytes.Buffer
	c.Stdin = &stdin
	c.Stdout = &stdout
	if suppressStderr {
		var stderr bytes.Buffer
		c.Stderr = &stderr
	} else {
		c.Stderr = os.Stderr
	}

	if err := c.Run(); err != nil {
		return "", fmt.Errorf("failed to execute: systemctl %s: %w", c, err)
	}

	return stdout.String(), nil
}
