package osutil

import (
	"context"
	"errors"
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"strings"
	"syscall"

	"golang.org/x/sys/windows/registry"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/security/skotty/skotty/internal/config"
	"a.yandex-team.ru/security/skotty/skotty/internal/paths"
	"a.yandex-team.ru/security/skotty/skotty/pkg/skottyctl"
)

const regKey = "SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Run"

func CheckService() error {
	k, err := registry.OpenKey(registry.CURRENT_USER, regKey, registry.READ)
	if err != nil {
		return err
	}
	defer func() { _ = k.Close() }()

	_, _, err = k.GetStringValue("Skotty")
	if errors.Is(err, registry.ErrNotExist) {
		return ErrNotInstalled
	}
	return err
}

func Restart() error {
	cfgPath, err := paths.Config()
	if err != nil {
		return fmt.Errorf("unable to get config path: %w", err)
	}

	cfg, err := config.Load(cfgPath, true)
	if err != nil {
		return ErrNotInstalled
	}

	ctl, err := skottyctl.NewClient(cfg.CtlSocketPath)
	if err != nil {
		return fmt.Errorf("unable to create skottyctl client: %w", err)
	}
	defer func() { _ = ctl.Close() }()

	if err := ctl.Restart(context.Background()); err != nil {
		if s, ok := status.FromError(err); ok && s.Code() == codes.Unimplemented {
			// backward compatibility
			return ErrNotSupported
		}
		return err
	}

	return nil
}

func InstallService() (string, error) {
	exePath, err := Executable()
	if err != nil {
		return "", fmt.Errorf("skotty executable not found: %w", err)
	}

	confDir, err := os.UserConfigDir()
	if err != nil {
		return "", fmt.Errorf("can't determine user config dir: %w", err)
	}

	err = createShortcut(exePath, filepath.Join(confDir, "Microsoft", "Windows", "Start Menu", "Programs", "Skotty.lnk"))
	if err != nil {
		return "", fmt.Errorf("can't create program link: %w", err)
	}

	k, err := registry.OpenKey(registry.CURRENT_USER, regKey, registry.SET_VALUE)
	if err != nil {
		return "", err
	}
	defer func() { _ = k.Close() }()

	err = k.SetStringValue("Skotty", fmt.Sprintf(`"%s" start --daemonize --tray`, exePath))
	if err != nil {
		return "", err
	}

	cmd := exec.Command(exePath, "start", "--daemonize", "--tray")
	cmd.SysProcAttr = &syscall.SysProcAttr{
		HideWindow: true,
	}
	if err = cmd.Run(); err != nil {
		return "skotty start --daemonize --tray", nil
	}

	return "", nil
}

func UnInstallService() error {
	confDir, err := os.UserConfigDir()
	if err != nil {
		return fmt.Errorf("can't determine user config dir: %w", err)
	}
	_ = os.Remove(filepath.Join(confDir, "Microsoft", "Windows", "Start Menu", "Programs", "Skotty.lnk"))

	k, err := registry.OpenKey(registry.CURRENT_USER, regKey, registry.SET_VALUE)
	if err != nil {
		return err
	}
	defer func() { _ = k.Close() }()

	return k.DeleteValue("Skotty")
}

func createShortcut(exePath, target string) error {
	const createShortcutShellCmdFmt = `
		$ProgramLocation = "%s"
		$ProgramArguments = "%s"
		$ShortcutLocation = "%s"
		$WScriptShell = New-Object -ComObject WScript.Shell
		$Shortcut = $WScriptShell.CreateShortcut($ShortcutLocation)
		$Shortcut.TargetPath = $ProgramLocation
		$Shortcut.Arguments = $ProgramArguments
		$Shortcut.WindowStyle = 7
		$Shortcut.Save()
	`

	script := fmt.Sprintf(createShortcutShellCmdFmt, escapePowershell(exePath), "start --daemonize --tray", escapePowershell(target))
	if err := runScript([]byte(script)); err != nil {
		return err
	}

	return nil
}

func runScript(content []byte) error {
	tmpfile, err := os.CreateTemp("", "skotty-osutil-*.ps1")
	if err != nil {
		return fmt.Errorf("create script: %w", err)
	}
	defer func() {
		if err != nil {
			return
		}

		_ = os.Remove(tmpfile.Name())
	}()

	if _, err = tmpfile.Write(content); err != nil {
		return fmt.Errorf("write script: %w", err)
	}

	if err = tmpfile.Close(); err != nil {
		return fmt.Errorf("close script: %w", err)
	}

	cmd := exec.Command("PowerShell", "–NoLogo", "–NoProfile", "-ExecutionPolicy", "Bypass", "-File", tmpfile.Name())
	cmd.SysProcAttr = &syscall.SysProcAttr{
		HideWindow: true,
	}
	if err = cmd.Run(); err != nil {
		return fmt.Errorf("run service script %q: %w", cmd, err)
	}

	return nil
}

func escapePowershell(in string) string {
	return strings.ReplaceAll(in, "\"", "`\"")
}
