package skottyctl

import (
	"context"
	"fmt"
	"os"

	"google.golang.org/grpc"

	"a.yandex-team.ru/security/skotty/skotty/internal/stdenv"
	"a.yandex-team.ru/security/skotty/skotty/pkg/skottyctl/skottyrpc"
)

type Client struct {
	addr string
	conn *grpc.ClientConn
	sc   skottyrpc.SupervisorClient
}

func NewClient(addr string) (client *Client, err error) {
	client = &Client{
		addr: addr,
	}
	client.conn, err = NewConnection(client.addr)
	if err != nil {
		return nil, fmt.Errorf("failed to create gRPC connection: %w", err)
	}
	client.sc = skottyrpc.NewSupervisorClient(client.conn)
	return client, nil
}

func (c *Client) Close() error {
	if err := c.conn.Close(); err != nil {
		return fmt.Errorf("failed to close gRPC connection: %w", err)
	}
	return nil
}

func (c *Client) Status(ctx context.Context) (*StatusReply, error) {
	rsp, err := c.sc.Status(ctx, &skottyrpc.StatusRequest{})
	if err != nil {
		return nil, err
	}

	return &StatusReply{
		Status:  rsp.Status,
		Version: rsp.Version,
		PID:     int(rsp.Pid),
	}, nil
}

func (c *Client) ReloadKeys(ctx context.Context) error {
	_, err := c.sc.ReloadKeys(ctx, &skottyrpc.ReloadKeysRequest{})
	return err
}

func (c *Client) Restart(ctx context.Context) error {
	_, err := c.sc.Restart(ctx, &skottyrpc.RestartRequest{})
	return err
}

func (c *Client) UpdateStartupTTY(ctx context.Context) error {
	env := make([]*skottyrpc.Env, 0, len(stdenv.StartupEnvs))
	for _, name := range stdenv.StartupEnvs {
		if val, ok := os.LookupEnv(name); ok {
			env = append(env, &skottyrpc.Env{
				Name:  name,
				Value: val,
			})
		}
	}

	_, err := c.sc.UpdateStartupTTY(ctx, &skottyrpc.UpdateStartupTTYRequest{
		Env: env,
	})
	return err
}
