package softattest

import (
	"crypto/x509"
	"crypto/x509/pkix"
	"encoding/asn1"
	"fmt"
)

type Attestation struct {
	Serial      string
	PINPolicy   PINPolicy
	TouchPolicy TouchPolicy
}

func Verify(attestationCert, slotCert *x509.Certificate) (*Attestation, error) {
	v := verifier{
		root: SharedAttestator().Public,
	}
	return v.Verify(attestationCert, slotCert)
}

type verifier struct {
	root *x509.Certificate
}

func (v *verifier) Verify(attestationCert, slotCert *x509.Certificate) (*Attestation, error) {
	if err := verifySignature(v.root, attestationCert); err != nil {
		return nil, fmt.Errorf("attestation certifcate not signed by : %v", err)
	}

	if err := verifySignature(attestationCert, slotCert); err != nil {
		return nil, fmt.Errorf("slot certificate not signed by attestation certifcate: %v", err)
	}
	return parseAttestation(slotCert)
}

func parseAttestation(slotCert *x509.Certificate) (*Attestation, error) {
	var a Attestation

	parseExt := func(e pkix.Extension) error {
		switch {
		case e.Id.Equal(extIDKeyPolicy):
			if len(e.Value) != 2 {
				return fmt.Errorf("expected 2 bytes from key policy, got: %d", len(e.Value))
			}
			switch e.Value[0] {
			case 0x01:
				a.PINPolicy = PINPolicyNever
			case 0x02:
				a.PINPolicy = PINPolicyOnce
			case 0x03:
				a.PINPolicy = PINPolicyAlways
			default:
				return fmt.Errorf("unrecognized pin policy: 0x%x", e.Value[0])
			}
			switch e.Value[1] {
			case 0x01:
				a.TouchPolicy = TouchPolicyNever
			case 0x02:
				a.TouchPolicy = TouchPolicyCached
			case 0x03:
				a.TouchPolicy = TouchPolicyAlways
			default:
				return fmt.Errorf("unrecognized touch policy: 0x%x", e.Value[1])
			}
		case e.Id.Equal(extIDSerialNumber):
			if _, err := asn1.Unmarshal(e.Value, &a.Serial); err != nil {
				return fmt.Errorf("parsing serial number: %v", err)
			}
		}

		return nil
	}

	for _, ext := range slotCert.Extensions {
		if err := parseExt(ext); err != nil {
			return nil, fmt.Errorf("parsing extension: %v", err)
		}
	}
	return &a, nil
}

func verifySignature(parent, c *x509.Certificate) error {
	return parent.CheckSignature(c.SignatureAlgorithm, c.RawTBSCertificate, c.Signature)
}
