package sshclient

import (
	"fmt"
	"os/exec"
	"path/filepath"
	"runtime"
	"strings"

	"github.com/blang/semver/v4"
	"github.com/mitchellh/go-homedir"
)

type ClientKind uint8

const (
	ClientKindNone ClientKind = iota
	ClientKindOpenSSH
	ClientKindWin32OpenSSH
	ClientKindCygwin
	ClientKindWSL
	ClientPutty
)

func (c ClientKind) String() string {
	switch c {
	case ClientKindOpenSSH:
		return "OpenSSH"
	case ClientKindWin32OpenSSH:
		return "Win32-OpenSSH"
	case ClientPutty:
		return "Putty"
	case ClientKindCygwin:
		return "Cygwin"
	case ClientKindWSL:
		return "WSL"
	default:
		return fmt.Sprintf("client_%d", c)
	}
}

func (c ClientKind) nativeName() string {
	switch c {
	case ClientKindOpenSSH:
		return "OpenSSH"
	case ClientKindWin32OpenSSH:
		return "OpenSSH_for_Windows"
	default:
		return fmt.Sprintf("client_%d", c)
	}
}

func (c ClientKind) SocketName(baseName string) string {
	switch c {
	case ClientKindOpenSSH:
		return baseName
	default:
		return fmt.Sprintf("%s-%s", baseName, strings.ToLower(c.String()))
	}
}

func (c ClientKind) ConfigPath() (string, error) {
	switch c {
	case ClientKindOpenSSH, ClientKindWin32OpenSSH:
		homeDir, err := homedir.Dir()
		if err != nil {
			return "", fmt.Errorf("unable to determine home dir: %w", err)
		}
		return filepath.Join(homeDir, ".ssh", "config"), nil
	default:
		return "", fmt.Errorf("not implemented for client %s", c.String())
	}
}

func (c ClientKind) RawVersion() (string, error) {
	cmd := exec.Command("ssh", "-V")
	var stdout, stderr strings.Builder
	cmd.Stdout = &stdout
	cmd.Stderr = &stderr

	if err := cmd.Run(); err != nil {
		return "", fmt.Errorf("can't execute 'ssh -V': %w", err)
	}

	return stderr.String(), nil
}

func (c ClientKind) Version() (semver.Version, error) {
	raw, err := c.RawVersion()
	if err != nil {
		return semver.Version{}, err
	}

	return parseSSHVersion(c.nativeName(), raw)
}

func BestClient() ClientKind {
	switch runtime.GOOS {
	case "windows":
		return ClientKindWin32OpenSSH
	default:
		return ClientKindOpenSSH
	}
}
