package config

import (
	"context"
	"errors"
	"os"
	"os/user"
	"path/filepath"
	"strings"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/file"
	"github.com/mitchellh/go-homedir"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/skotty/wsl2proxy/internal/listener"
)

type Socket struct {
	EnvName     string `yaml:"env_name"`
	LinuxAddr   string `yaml:"linux_addr"`
	WindowsAddr string `yaml:"windows_addr"`
}

type Config struct {
	LogLevel   log.Level `yaml:"log_level"`
	LogPath    string    `yaml:"log_path"`
	RuntimeDir string    `yaml:"runtime_dir"`
	Sockets    []Socket  `yaml:"sockets"`
}

func (c *Config) Build() error {
	if len(c.Sockets) == 0 {
		return errors.New("no sockets configured")
	}

	for i, s := range c.Sockets {
		c.Sockets[i].LinuxAddr = strings.ReplaceAll(s.LinuxAddr, "{runtime_dir}", c.RuntimeDir)
	}

	c.LogPath = strings.ReplaceAll(c.LogPath, "{runtime_dir}", c.RuntimeDir)
	return nil
}

func (c *Config) ListenersSpec() listener.Spec {
	spec := listener.Spec{
		Pairs: make([]listener.ListenPair, len(c.Sockets)),
	}
	for i, sock := range c.Sockets {
		spec.Pairs[i] = listener.ListenPair{
			Name: sock.EnvName,
			Src:  sock.LinuxAddr,
			Dst:  sock.WindowsAddr,
		}
	}

	return spec
}

func Load(cfgPath string) (*Config, error) {
	runtimeDir := os.TempDir()
	if xdg := os.Getenv("XDG_RUNTIME_DIR"); xdg != "" {
		runtimeDir = xdg
	}

	username := "na"
	if u, err := user.Current(); err == nil {
		username = u.Username
	}

	runtimeDir = filepath.Join(runtimeDir, "wsl2proxy-"+username)
	out := &Config{
		LogLevel:   log.InfoLevel,
		LogPath:    filepath.Join("{runtime_dir}", "proxy.log"),
		RuntimeDir: runtimeDir,
		Sockets: []Socket{
			{
				EnvName:     "SSH_AUTH_SOCK",
				LinuxAddr:   filepath.Join("{runtime_dir}", "default-ssh-auth.sock"),
				WindowsAddr: "pipe://./pipe/openssh-ssh-agent",
			},
			{
				EnvName:     "SSH_SUDO_AUTH_SOCK",
				LinuxAddr:   filepath.Join("{runtime_dir}", "sudo-ssh-auth.sock"),
				WindowsAddr: "pipe://./pipe/sudo-openssh-ssh-agent",
			},
		},
	}

	if cfgPath == "" {
		xdgPath, err := homedir.Expand("~/.config/wsl2proxy/config.yaml")
		if err != nil {
			if _, err := os.Stat(xdgPath); err == nil {
				cfgPath = xdgPath
			}
		}
	}

	if cfgPath != "" {
		loader := confita.NewLoader(file.NewBackend(cfgPath))
		if err := loader.Load(context.Background(), out); err != nil {
			return nil, err
		}
	}

	return out, out.Build()
}
