package pipeproxy

import (
	"fmt"
	"io"
	"os"
	"time"

	"github.com/hashicorp/yamux"
)

type Session struct {
	realSession *yamux.Session
}

type Stream interface {
	io.ReadWriteCloser
}

func NewSession(conn io.ReadWriteCloser) (*Session, error) {
	session, err := yamux.Client(conn, yamuxConfig())
	if err != nil {
		return nil, fmt.Errorf("unable to create yamux client: %w", err)
	}

	return &Session{
		realSession: session,
	}, nil
}

func (s *Session) OpenStream() (Stream, error) {
	s.realSession.IsClosed()
	return s.realSession.OpenStream()
}

func (s *Session) AcceptStream() (Stream, error) {
	return s.realSession.AcceptStream()
}

func (s *Session) IsAlive() bool {
	return !s.realSession.IsClosed()
}

func (s *Session) Close() error {
	return s.realSession.Close()
}

func yamuxConfig() *yamux.Config {
	return &yamux.Config{
		AcceptBacklog:          256,
		EnableKeepAlive:        true,
		KeepAliveInterval:      30 * time.Second,
		ConnectionWriteTimeout: 10 * time.Second,
		MaxStreamWindowSize:    256 * 1024,
		StreamCloseTimeout:     100 * time.Millisecond,
		StreamOpenTimeout:      5 * time.Second,
		LogOutput:              os.Stderr,
	}
}
