import csv
import importlib
import sys
from requests import Session
from requests.auth import HTTPBasicAuth

from zeep import Client
from zeep.transports import Transport

from security.soc.soc_modules.hec_sender.hec_sender import SplunkHECSender


def config(key=None):
    if 'sccm_fetcher' not in config.__dict__:
        sys.path.append('/etc/yandex')
        config.sccm_fetcher = importlib.import_module('vault').sccm_fetcher
    return config.sccm_fetcher[key] if key else config.sccm_fetcher


def get_report_from_sccm(report, client):
    report_path = report['report_path']
    params = []
    if report['params']:
        for k, v in report['params'].items():
            params.append(client.service.ParameterValue(k, v))
    params = client.service.ArrayOfParameterValue(params)
    result = client.service.LoadReport(Report=report_path)
    client.service.SetExecutionParameters(params, 'en-us', _soapheaders=[result.header])
    data = client.service.Render(Format='CSV', _soapheaders=[result.header])
    return data


def parse_native_report(data, report_name):
    data = data.replace('\r', '').strip().split('\n\n')[-1]
    data = data.split('\n')
    if len(data) == 0:
        return None
    headers_num = len(data[0].split(',')) // 2
    data = data[1:]
    headers = [header.replace(' ', '_') for header in data[0].split(',')]

    parsed_data = []
    for line in csv.reader(data):
        if not line:
            continue
        record = {'report_name': report_name}
        for i in range(headers_num):
            record[map_field(headers[i])] = line[headers_num+i].strip()
        parsed_data.append(record)

    return parsed_data


def parse_custom_report(data, report_name):
    data = data.replace('\r', '').strip().split('\n\n')[-1]
    data = data.split('\n')
    if len(data) == 0:
        return None
    headers_num = len(data[0].split(','))
    headers = [header.replace(' ', '_') for header in data[0].split(',')]
    data = data[1:]

    parsed_data = []
    for line in csv.reader(data):
        if not line:
            continue
        record = {'report_name': report_name}
        for i in range(headers_num):
            record[map_field(headers[i])] = line[i].strip()
        parsed_data.append(record)

    return parsed_data


def map_field(name):
    field = config('cim_mapping').get(name) or name
    return field.strip()


def main():
    hec = SplunkHECSender(token=config('hec_token'), current_time_as_timestamp=True, hec_verify_ssl=False)
    session = Session()
    session.auth = HTTPBasicAuth(config('username'), config('password'))
    client = Client('https://{}/ReportServer/ReportExecution2005.asmx?wsdl'.format(config('sccm_host')),
                    transport=Transport(session=session))

    client.service.ArrayOfParameterValue = client.get_type('ns0:ArrayOfParameterValue')
    client.service.ParameterValue = client.get_type('ns0:ParameterValue')
    for report in config('reports'):
        result = get_report_from_sccm(report, client)
        if report.get('custom'):
            data = parse_custom_report(result.body.Result.decode("utf-8"), report.get('report_name'))
        else:
            data = parse_native_report(result.body.Result.decode("utf-8"), report.get('report_name'))
        if data:
            hec.send_data(data)


if __name__ == '__main__':
    main()
