package commands

import (
	"context"
	"encoding/json"
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/soc/ypper/internal/yputil"
	"a.yandex-team.ru/yp/go/yp"
)

func init() {
	resolveCmd.AddCommand(resolvePodSetCmd)
	rootCmd.AddCommand(resolveCmd)
}

var resolveCmd = &cobra.Command{
	Use:          "resolve",
	SilenceUsage: true,
	Short:        "resolve YP objects",
}

type PodSetInfo struct {
	ID          string   `json:"id"`
	StageID     string   `json:"stage_id"`
	ProjectID   string   `json:"project_id"`
	IsSox       bool     `json:"is_sox"`
	Maintainers []string `json:"maintainers"`
}

var resolvePodSetCmd = &cobra.Command{
	Use:          "pod_set",
	SilenceUsage: true,
	Short:        "resolve pod_set",
	RunE: func(_ *cobra.Command, args []string) error {
		if len(args) == 0 {
			return xerrors.New("no podset_ids provided")
		}

		ypc, err := yp.NewClient("xdc", yp.WithLogger(logger), yp.WithSystemAuthToken())
		if err != nil {
			return xerrors.Errorf("failed to create YP-client: %w", err)
		}
		defer ypc.Close()

		out := json.NewEncoder(os.Stdout)
		ctx := context.Background()
		for _, podSetID := range args {
			l := log.With(logger, log.String("podset_id", podSetID))
			stageID, err := yputil.PodSetToStage(podSetID)
			if err != nil {
				l.Error("failed to parse stage", log.Error(err))
				continue
			}

			rsp, err := ypc.GetStage(ctx, yp.GetStageRequest{
				ID:     stageID,
				Format: yp.PayloadFormatYson,
				Selectors: []string{
					"/spec/sox_service",
					"/meta/project_id",
				},
			})

			if err != nil {
				l.Error("failed to lookup stage in YP-master", log.Error(err))
				continue
			}

			var (
				isSox     *bool
				projectID *string
			)
			if err := rsp.Fill(&isSox, &projectID); err != nil {
				l.Error("failed to parse stage info", log.Error(err))
				continue
			}

			if projectID == nil || *projectID == "" {
				l.Error("no project id", log.Error(err))
				continue
			}

			stageInfo := PodSetInfo{
				ID:        podSetID,
				StageID:   stageID,
				ProjectID: *projectID,
				IsSox:     isSox != nil && *isSox,
			}

			mr := yputil.NewMemberResolver(ypc, l)
			stageInfo.Maintainers, err = mr.StageMaintainers(ctx, stageInfo.ProjectID, stageInfo.StageID)
			if err != nil {
				l.Error("failed to resolve stage members", log.Error(err))
				continue
			}

			if err := out.Encode(stageInfo); err != nil {
				l.Error("failed to marshal results", log.Error(err))
				continue
			}
		}
		return nil
	},
}
