package yputil

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/karlseguin/ccache/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/yp/go/yp"
	"a.yandex-team.ru/yp/go/yp/yperrors"
	"a.yandex-team.ru/yt/go/yterrors"
)

const (
	cacheTTL = 1 * time.Hour
)

var cache = ccache.New(
	ccache.Configure().MaxSize(2048),
)

type MemberResolver struct {
	ypc *yp.Client
	l   log.Logger
}

func NewMemberResolver(ypc *yp.Client, l log.Logger) *MemberResolver {
	return &MemberResolver{
		ypc: ypc,
		l:   l,
	}
}

func (r *MemberResolver) StageMaintainers(ctx context.Context, projectID, stageID string) ([]string, error) {
	groupID := fmt.Sprintf("deploy:%s.%s.MAINTAINER", projectID, stageID)
	members, err := r.GroupMembers(ctx, groupID)
	if err != nil {
		return nil, xerrors.Errorf("failed to resolve group %s members: %w", groupID, err)
	}

	if len(members) > 0 {
		return r.ExpandMembers(ctx, members)
	}

	// Для стейджа нет роли MAINTAINER - в этом случае через объкт group резолвим список MAINTAINERs для проекта
	r.l.Info("no stage maintainers, try project maintainers", log.String("group_id", groupID))
	return r.ProjectMaintainers(ctx, projectID)
}

func (r *MemberResolver) ProjectMaintainers(ctx context.Context, projectID string) ([]string, error) {
	groupID := fmt.Sprintf("deploy:%s.MAINTAINER", projectID)
	members, err := r.GroupMembers(ctx, groupID)
	if err != nil {
		return nil, xerrors.Errorf("failed to resolve group members: %w", err)
	}

	if len(members) > 0 {
		return r.ExpandMembers(ctx, members)
	}

	// Для проекта нет роли MAINTAINER - в этом случае через объкт group резолвим список OWNERs для проекта
	r.l.Info("no project maintainers, try project owners", log.String("group_id", groupID))
	return r.ProjectOwners(ctx, projectID)
}

func (r *MemberResolver) ProjectOwners(ctx context.Context, projectID string) ([]string, error) {
	groupID := fmt.Sprintf("deploy:%s.OWNER", projectID)
	members, err := r.GroupMembers(ctx, groupID)
	if err != nil {
		return nil, xerrors.Errorf("failed to resolve group members: %w", err)
	}

	if len(members) > 0 {
		return r.ExpandMembers(ctx, members)
	}

	r.l.Info("no project owners, nothing to try else", log.String("group_id", groupID))
	return nil, nil
}

func (r *MemberResolver) GroupMembers(ctx context.Context, groupID string) ([]string, error) {
	item, err := cache.Fetch(groupID, cacheTTL, func() (interface{}, error) {
		var members []string
		_, err := r.ypc.GetGroupAttr(
			ctx,
			yp.GetGroupAttrRequest{
				ID:       groupID,
				Selector: "/spec/members",
			},
			&members,
		)

		if yterrors.ContainsErrorCode(err, yperrors.CodeNoSuchObject) {
			// this is fine, just no group id
			return nil, nil
		}

		if err != nil {
			return nil, xerrors.Errorf("failed to lookup group members: %w", err)
		}
		return members, nil
	})

	if err != nil {
		return nil, err
	}

	v := item.Value()
	if v == nil {
		return nil, nil
	}

	return r.ExpandMembers(ctx, v.([]string))
}

func (r *MemberResolver) ExpandMembers(ctx context.Context, members []string) ([]string, error) {
	out := make([]string, 0, len(members))
	for _, member := range members {
		if !strings.HasPrefix(member, "idm:") {
			out = append(out, member)
			continue
		}

		groupMembers, err := r.GroupMembers(ctx, member)
		if err != nil {
			r.l.Error("failed to resolve member group", log.String("group_id", member))
			continue
		}

		out = append(out, groupMembers...)
	}
	return out, nil
}
