package app

import (
	"context"
	"fmt"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/security/libs/go/xlock"
	"a.yandex-team.ru/security/ssh-exporter/internal/config"
	"a.yandex-team.ru/security/ssh-exporter/internal/exportdb"
	"a.yandex-team.ru/security/ssh-exporter/internal/models"
	"a.yandex-team.ru/security/ssh-exporter/internal/nanny"
	"a.yandex-team.ru/security/ssh-exporter/internal/splunk"
	"a.yandex-team.ru/security/ssh-exporter/internal/staff"
	"a.yandex-team.ru/security/ssh-exporter/internal/yp"
	"a.yandex-team.ru/yt/go/yterrors"
)

const (
	batchLimit = 2000
)

type App struct {
	tickInternal time.Duration
	exportDB     *exportdb.DB
	ypc          *yp.Client
	staffc       *staff.Client
	nannyc       *nanny.Client
	splunkc      *splunk.Client
	locker       xlock.Locker
	cfg          *config.Config
	log          log.Logger
	lastSync     time.Time
	ctx          context.Context
	cancelCtx    context.CancelFunc
	closed       chan struct{}
}

func NewApp(cfg *config.Config) (*App, error) {
	logLvl := log.InfoLevel
	if cfg.Dev {
		logLvl = log.DebugLevel
	}

	logger, err := zap.NewDeployLogger(logLvl)
	if err != nil {
		return nil, fmt.Errorf("create logger: %w", err)
	}

	locker, err := xlock.NewYTLocker(cfg.YT.Proxy, cfg.YT.Path, cfg.YT.Token)
	if err != nil {
		return nil, fmt.Errorf("create YT locker: %w", err)
	}

	ypc, err := yp.NewClient("xdc", yp.WithAuthToken(cfg.YP.Token), yp.WithLogger(logger))
	if err != nil {
		return nil, fmt.Errorf("create YP client: %w", err)
	}

	staffc, err := staff.NewClient(staff.WithAuthToken(cfg.Staff.Token))
	if err != nil {
		return nil, fmt.Errorf("create staff client: %w", err)
	}

	splunkc := splunk.NewClient(splunk.WithUpstream(cfg.Splunk.Upstream))
	nannyc := nanny.NewClient(cfg.Nanny.Token)

	exportDB, err := exportdb.NewDB(context.TODO(), cfg.ExportDB)
	if err != nil {
		return nil, fmt.Errorf("create export db: %w", err)
	}

	ctx, cancel := context.WithCancel(context.Background())
	return &App{
		tickInternal: cfg.SyncPeriod,
		exportDB:     exportDB,
		ypc:          ypc,
		nannyc:       nannyc,
		splunkc:      splunkc,
		staffc:       staffc,
		cfg:          cfg,
		locker:       locker,
		//lastSync:     time.Now().Add(-24 * time.Hour * 90),
		log:       logger,
		ctx:       ctx,
		cancelCtx: cancel,
		closed:    make(chan struct{}),
	}, nil
}

func (a *App) Start() error {
	defer close(a.closed)

	tickInterval := a.tickInternal
	for {
		toNextWork := time.Until(
			time.Now().Add(tickInterval).Truncate(tickInterval),
		)

		a.log.Infof("%s to next sync", toNextWork)
		select {
		case <-a.ctx.Done():
			return nil
		case <-time.After(toNextWork):
			if err := a.Export(); err != nil {
				a.log.Error("process failed, will sleep 5 minutes", log.Error(err))
				tickInterval = 5 * time.Minute
				continue
			}

			tickInterval = a.tickInternal
		}
	}
}

func (a *App) Export() error {
	a.log.Info("export YP stages")
	tx, err := a.locker.Lock(a.ctx)
	if err != nil {
		if yterrors.ContainsErrorCode(err, yterrors.CodeConcurrentTransactionLockConflict) {
			a.log.Info("conflict YT lock", log.String("error", err.Error()))
			return nil
		}

		return fmt.Errorf("lock acquire: %w", err)
	}
	defer func() { _ = tx.Unlock() }()

	if err := a.syncSSHUsages(); err != nil {
		return fmt.Errorf("unable to sync SSH usages: %w", err)
	}

	if err := a.syncDictionaries(); err != nil {
		return fmt.Errorf("unable to sync dictionaries usages: %w", err)
	}

	return nil
}

func (a *App) syncSSHUsages() error {
	if err := a.splunkc.Login(a.ctx, a.cfg.Splunk.Username, a.cfg.Splunk.Password); err != nil {
		return fmt.Errorf("unable to login into splunk: %w", err)
	}

	syncTo := time.Now().Truncate(a.tickInternal)
	syncFrom := a.lastSync
	if syncFrom.IsZero() {
		syncFrom = syncTo.Add(-a.tickInternal)
	}

	for syncFrom.Before(syncTo) {
		from := syncFrom.Truncate(a.tickInternal)
		to := syncFrom.Add(a.tickInternal).Round(a.tickInternal)

		a.log.Info("sync SSH usages", log.Time("from", from), log.Time("to", to))
		splunkUsages, err := a.splunkc.SSHUsages(a.ctx, from, to)
		if err != nil {
			return fmt.Errorf("unable to get usages for interval %s - %s: %w", from, to, err)
		}

		a.log.Infof("splunk returns %d usages", len(splunkUsages))
		syncFrom = to.Truncate(a.tickInternal)
		if len(splunkUsages) == 0 {
			continue
		}

		usages := make([]models.SSHUsage, 0, batchLimit)
		usageCount := 0
		for _, usage := range splunkUsages {
			usages = append(usages, models.SSHUsage{
				StaffUser:  usage.SrcUser,
				TargetUser: usage.User,
				SyncTime:   to,
				SystemID:   systemIDFromSplunk(usage.NannyService, usage.DeployStage),
				Count:      usage.Count,
			})

			usageCount++
			if usageCount < batchLimit {
				continue
			}

			if err := a.exportDB.InsertSSHUsage(a.ctx, usages); err != nil {
				return fmt.Errorf("unable to save SSH usages: %w", err)
			}
			usageCount = 0
			usages = usages[:0]
		}

		if err := a.exportDB.InsertSSHUsage(a.ctx, usages); err != nil {
			return fmt.Errorf("unable to save SSH usages: %w", err)
		}
	}

	return nil
}

func (a *App) syncDictionaries() error {
	a.log.Info("export staff users")
	usersMap := make(map[string]uint64)
	groupsMap := make(map[int][]string)
	err := a.staffc.WalkUsers(a.ctx, func(user staff.User) error {
		usersMap[user.Login] = user.UID
		for _, g := range user.Groups {
			groupsMap[g] = append(groupsMap[g], user.Login)
		}
		return nil
	})
	if err != nil {
		return fmt.Errorf("unable to export staff users: %w", err)
	}

	a.log.Infof("staff returns %d users", len(usersMap))
	a.log.Infof("staff returns %d groups", len(groupsMap))

	if err := a.exportDB.DropTables(a.ctx); err != nil {
		return fmt.Errorf("unable to drop temporary tables: %w", err)
	}

	if err := a.exportDB.CreateCommonTables(a.ctx); err != nil {
		return fmt.Errorf("unable to create temporary tables: %w", err)
	}

	syncYP := func() error {
		a.log.Info("export YP stages")
		ypStages, err := a.ypc.ListStages(a.ctx)
		if err != nil {
			return fmt.Errorf("unable to list YP stages: %w", err)
		}

		permsCount := 0
		aclCount := 0
		projectsCount := 0
		var permsToSync []models.SSHPermission
		var aclToSync []models.RowACL
		var projectsToSync []models.Project

		exportToDB := func() error {
			if err := a.exportDB.InsertProjects(a.ctx, projectsToSync); err != nil {
				return fmt.Errorf("unable to sync projects: %w", err)
			}

			if err := a.exportDB.InsertSSHPermissions(a.ctx, permsToSync); err != nil {
				return fmt.Errorf("unable to sync SSH permissions: %w", err)
			}

			if err := a.exportDB.InsertACL(a.ctx, aclToSync); err != nil {
				return fmt.Errorf("unable to sync row ACL: %w", err)
			}

			projectsCount = 0
			projectsToSync = projectsToSync[:0]
			permsCount = 0
			permsToSync = permsToSync[:0]
			aclCount = 0
			aclToSync = aclToSync[:0]
			return nil
		}

		for _, stage := range ypStages {
			systemID := fmt.Sprintf("deploy:%s", stage.ID)
			projectsToSync = append(projectsToSync, models.Project{
				SystemID: systemID,
				Project:  fmt.Sprintf("deploy:%s", stage.Project),
			})
			projectsCount++

			seen := map[string]struct{}{}
			for _, u := range stage.SSHUsers {
				if _, ok := seen[u]; ok {
					continue
				}

				seen[u] = struct{}{}
				permsToSync = append(permsToSync, models.SSHPermission{
					StaffUser: u,
					SystemID:  systemID,
				})
				permsCount++
			}

			for _, u := range stage.Writers {
				uid, ok := usersMap[u]
				if !ok {
					continue
				}

				aclToSync = append(aclToSync, models.RowACL{
					SystemID:  systemID,
					UserID:    uid,
					StaffUser: u,
				})
				aclCount++
			}

			if permsCount < batchLimit && aclCount < batchLimit && projectsCount < batchLimit {
				continue
			}

			if err := exportToDB(); err != nil {
				return err
			}
		}

		a.log.Info("done")
		return exportToDB()
	}

	syncNanny := func() error {
		a.log.Info("export Nanny services")
		services, err := a.nannyc.ListServices(a.ctx)
		if err != nil {
			return fmt.Errorf("unable to list nanny services: %w", err)
		}

		a.log.Infof("nanny returns %d services", len(services))

		permsCount := 0
		aclCount := 0
		projectsCount := 0
		var permsToSync []models.SSHPermission
		var aclToSync []models.RowACL
		var projectsToSync []models.Project

		a.log.Info("store Nanny services")
		appendPerms := func(systemID string, acl nanny.ACL) {
			for _, user := range acl.Logins {
				permsToSync = append(permsToSync, models.SSHPermission{
					SystemID:  systemID,
					StaffUser: user,
				})
				permsCount++
			}

			for _, group := range acl.Groups {
				gid, err := strconv.Atoi(group)
				if err != nil {
					continue
				}

				for _, user := range groupsMap[gid] {
					permsToSync = append(permsToSync, models.SSHPermission{
						SystemID:  systemID,
						StaffUser: user,
					})
					permsCount++
				}
			}
		}

		appendACL := func(systemID string, acl nanny.ACL) {
			for _, user := range acl.Logins {
				uid, ok := usersMap[user]
				if !ok {
					continue
				}

				aclToSync = append(aclToSync, models.RowACL{
					SystemID:  systemID,
					UserID:    uid,
					StaffUser: user,
				})
				aclCount++
			}

			for _, group := range acl.Groups {
				gid, err := strconv.Atoi(group)
				if err != nil {
					continue
				}

				for _, user := range groupsMap[gid] {
					uid, ok := usersMap[user]
					if !ok {
						continue
					}

					aclToSync = append(aclToSync, models.RowACL{
						SystemID:  systemID,
						UserID:    uid,
						StaffUser: user,
					})
					aclCount++
				}
			}
		}

		exportToDB := func() error {
			if err := a.exportDB.InsertProjects(a.ctx, projectsToSync); err != nil {
				return fmt.Errorf("unable to sync projects: %w", err)
			}

			if err := a.exportDB.InsertSSHPermissions(a.ctx, permsToSync); err != nil {
				return fmt.Errorf("unable to sync SSH permissions: %w", err)
			}

			if err := a.exportDB.InsertACL(a.ctx, aclToSync); err != nil {
				return fmt.Errorf("unable to sync row ACL: %w", err)
			}

			projectsCount = 0
			projectsToSync = projectsToSync[:0]
			permsCount = 0
			permsToSync = permsToSync[:0]
			aclCount = 0
			aclToSync = aclToSync[:0]
			return nil
		}

		for _, service := range services {
			if service.CurrentState.Content.Summary.Value != nanny.OnlineSummary {
				continue
			}

			systemID := fmt.Sprintf("nanny:%s", service.ID)
			projectsToSync = append(projectsToSync, models.Project{
				SystemID: systemID,
				Project:  fmt.Sprintf("nanny:%s", strings.TrimRight(service.InfoAttrs.Content.Category, "/")),
			})
			projectsCount++

			appendPerms(systemID, service.AuthAttrs.Content.Owners)
			appendPerms(systemID, service.AuthAttrs.Content.ConfManagers)

			appendACL(systemID, service.AuthAttrs.Content.Owners)
			appendACL(systemID, service.AuthAttrs.Content.OpsManagers)
			appendACL(systemID, service.AuthAttrs.Content.ConfManagers)

			if permsCount < batchLimit && aclCount < batchLimit && projectsCount < batchLimit {
				continue
			}

			if err := exportToDB(); err != nil {
				return err
			}
		}

		a.log.Info("done")
		return exportToDB()
	}

	if err := syncYP(); err != nil {
		return fmt.Errorf("unable to sync YP stages: %w", err)
	}

	if err := syncNanny(); err != nil {
		return fmt.Errorf("unable to sync Nanny services: %w", err)
	}

	return nil
}

func (a *App) Shutdown(ctx context.Context) error {
	a.cancelCtx()

	select {
	case <-ctx.Done():
		return ctx.Err()
	case <-a.closed:
		return nil
	}
}

func systemIDFromSplunk(nannyService, podSetID string) string {
	switch {
	case nannyService != "":
		return fmt.Sprintf("nanny:%s", nannyService)
	case podSetID != "":
		if idx := strings.IndexByte(podSetID, '.'); idx > 0 {
			return fmt.Sprintf("deploy:%s", podSetID[:idx])
		}
		return fmt.Sprintf("deploy:%s", podSetID)
	default:
		return "unknown"
	}
}
