package commands

import (
	"context"
	"errors"
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/ssh-exporter/internal/app"
	"a.yandex-team.ru/security/ssh-exporter/internal/config"
)

var startCmd = &cobra.Command{
	Use:          "start",
	SilenceUsage: true,
	Short:        "starts export",
	RunE: func(_ *cobra.Command, _ []string) error {
		if rootArgs.ConfigPath == "" {
			return errors.New("--config is empty")
		}

		cfg, err := config.Load(rootArgs.ConfigPath)
		if err != nil {
			return fmt.Errorf("failed to parse config: %w", err)
		}

		instance, err := app.NewApp(cfg)
		if err != nil {
			return fmt.Errorf("failed to create application: %w", err)
		}

		errChan := make(chan error, 1)
		okChan := make(chan struct{}, 1)
		go func() {
			if err := instance.Start(); err != nil {
				errChan <- fmt.Errorf("failed to start application: %w", err)
			} else {
				okChan <- struct{}{}
			}
		}()

		sigChan := make(chan os.Signal, 1)
		signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM)

		select {
		case <-sigChan:
			ctx, cancel := context.WithTimeout(context.Background(), config.ShutdownDeadline)
			defer cancel()

			return instance.Shutdown(ctx)
		case <-okChan:
			return nil
		case err := <-errChan:
			return err
		}
	},
}
