package config

import (
	"context"
	"os"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"
)

const ShutdownDeadline = 30 * time.Minute

type YDB struct {
	Endpoint string `yaml:"endpoint"`
	Database string `yaml:"database"`
	Path     string `yaml:"path"`
	Token    string `yaml:"token"`
}

type Splunk struct {
	Upstream string `yaml:"upstream"`
	Username string `yaml:"username"`
	Password string `yaml:"password"`
}

type YT struct {
	Proxy string `yaml:"proxy"`
	Path  string `yaml:"path"`
	Token string `yaml:"token"`
}

type YP struct {
	Token string `yaml:"token"`
}

type Nanny struct {
	Token string `yaml:"token"`
}

type Staff struct {
	Token string `yaml:"token"`
}

type Config struct {
	Dev        bool          `yaml:"dev"`
	SyncPeriod time.Duration `yaml:"sync_period"`
	ExportDB   YDB           `yaml:"export_db"`
	YP         YP            `yaml:"yp"`
	YT         YT            `yaml:"yt"`
	Splunk     Splunk        `yaml:"splunk"`
	Nanny      Nanny         `yaml:"nanny"`
	Staff      Staff         `yaml:"token"`
}

func Load(cfgPath string) (*Config, error) {
	cfg := &Config{
		YT: YT{
			Proxy: "locke",
			Path:  "//home/security/ssh-export/dev",
			Token: os.Getenv("YT_TOKEN"),
		},
		ExportDB: YDB{
			Token: os.Getenv("YDB_TOKEN"),
		},
		Staff: Staff{
			Token: os.Getenv("STAFF_TOKEN"),
		},
		Nanny: Nanny{
			Token: os.Getenv("NANNY_TOKEN"),
		},
		YP: YP{
			Token: os.Getenv("YP_TOKEN"),
		},
		Splunk: Splunk{
			Upstream: "https://search.splunk.yandex.net:8089",
			Password: os.Getenv("SPLUNK_PASSWORD"),
		},
		SyncPeriod: 1 * time.Hour,
	}

	loader := confita.NewLoader(env.NewBackend(), file.NewBackend(cfgPath))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, err
	}

	return cfg, nil
}
