package nanny

import (
	"context"
	"crypto/tls"
	"fmt"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
)

type Client struct {
	httpc *resty.Client
}

func NewClient(token string) *Client {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		panic(err)
	}

	httpc := resty.New().
		SetRetryCount(5).
		SetHeader("Authorization", "OAuth "+token).
		SetTLSClientConfig(&tls.Config{RootCAs: certPool}).
		AddRetryCondition(func(r *resty.Response, err error) bool {
			return !r.IsSuccess() || err != nil
		}).
		SetBaseURL("https://nanny.yandex-team.ru/")

	return &Client{
		httpc: httpc,
	}
}

func (c *Client) ListServices(ctx context.Context) ([]Service, error) {
	var results struct {
		Result []Service `json:"result"`
	}

	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetResult(&results).
		Get("/v2/services/?exclude_runtime_attrs=true")
	if err != nil {
		return nil, fmt.Errorf("request failed: %w", err)
	}

	if !rsp.IsSuccess() {
		return nil, fmt.Errorf("request failed: %s", rsp.Status())
	}

	return results.Result, nil
}

func (c *Client) AuthAttrs(ctx context.Context, service string) (*AuthAttrs, error) {
	var out AuthAttrs
	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetResult(&out).
		SetPathParam("service", service).
		Get("/v2/services/{service}/auth_attrs/")
	if err != nil {
		return nil, fmt.Errorf("request failed: %w", err)
	}

	if !rsp.IsSuccess() {
		return nil, fmt.Errorf("request failed: %s", rsp.Status())
	}

	return &out, nil
}
