package splunk

import (
	"context"
	"crypto/tls"
	"errors"
	"fmt"
	"io"
	"strconv"
	"time"

	"github.com/go-resty/resty/v2"
	"github.com/valyala/fastjson"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/httputil/headers"
)

const (
	DefaultUpstream  = "https://search.splunk.yandex.net:8089"
	splunkTimeFormat = "01/02/2006:15:04:05"
)

type Client struct {
	httpc      *resty.Client
	sessionKey string
}

func NewClient(opts ...Option) *Client {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		panic(fmt.Sprintf("unable to create certifi pool: %v", err))
	}

	httpc := resty.New().
		SetRetryCount(3).
		SetTLSClientConfig(&tls.Config{RootCAs: certPool, InsecureSkipVerify: true}).
		SetBaseURL(DefaultUpstream).
		SetHeader(
			headers.UserAgentKey,
			fmt.Sprintf("Go Splunk client r%s", buildinfo.Info.ArcadiaSourceRevision),
		)

	c := &Client{
		httpc: httpc,
	}

	for _, opt := range opts {
		opt(c)
	}

	return &Client{
		httpc: httpc,
	}
}

func (c *Client) Login(ctx context.Context, username, password string) error {
	var sessionInfo struct {
		SessionKey string `json:"sessionKey"`
	}
	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetResult(&sessionInfo).
		SetFormData(map[string]string{
			"username":    username,
			"password":    password,
			"output_mode": "json",
		}).
		Post("/services/auth/login")

	if err != nil {
		return fmt.Errorf("unable to request login: %w", err)
	}

	if !rsp.IsSuccess() {
		return fmt.Errorf("non-200 response: %s", rsp.Status())
	}

	if sessionInfo.SessionKey == "" {
		return errors.New("splunk returns empty session key")
	}

	c.sessionKey = sessionInfo.SessionKey
	return nil
}

func (c *Client) Search(ctx context.Context, searchString string) (*fastjson.Scanner, error) {
	if c.sessionKey == "" {
		return nil, errors.New("no Splunk session key, you must to login first")
	}

	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetDoNotParseResponse(true).
		SetHeader("Authorization", "Splunk "+c.sessionKey).
		SetFormData(map[string]string{
			"search":      "search " + searchString,
			"output_mode": "json_rows"}).
		Post("/services/search/jobs/export")

	if err != nil {
		return nil, fmt.Errorf("unable to request: %w", err)
	}

	defer func() { _ = rsp.RawBody().Close() }()
	body, err := io.ReadAll(rsp.RawBody())
	if err != nil {
		return nil, fmt.Errorf("unable to read body: %w", err)
	}

	if !rsp.IsSuccess() {
		return nil, fmt.Errorf("non-200 response: %s", string(body))
	}

	var sc fastjson.Scanner
	sc.InitBytes(body)
	return &sc, nil
}

func (c *Client) SSHUsages(ctx context.Context, from, to time.Time) ([]SSHUsage, error) {
	req := fmt.Sprintf(`index=sshd sourcetype=portoshell earliest=%q latest=%q
| fillnull service pod_set_id value=""
| stats count by src_user, user, service, pod_set_id`, from.Format(splunkTimeFormat), to.Format(splunkTimeFormat))
	sc, err := c.Search(ctx, req)
	if err != nil {
		return nil, fmt.Errorf("search failed: %w", err)
	}

	var out []SSHUsage
	for sc.Next() {
		data := sc.Value()
		if data.GetBool("preview") {
			continue
		}

		for _, row := range data.GetArray("rows") {
			values, err := row.Array()
			if err != nil {
				panic(err)
			}

			var sshUsage SSHUsage
			for i, value := range values {
				bytes, err := value.StringBytes()
				switch i {
				case 0:
					sshUsage.SrcUser = string(bytes)
				case 1:
					sshUsage.User = string(bytes)
				case 2:
					sshUsage.NannyService = string(bytes)
				case 3:
					sshUsage.DeployStage = string(bytes)
				case 4:
					var cnt uint64
					cnt, err = strconv.ParseUint(string(bytes), 10, 32)
					if err == nil {
						sshUsage.Count = uint32(cnt)
					}
				}

				if sshUsage.NannyService != "" && sshUsage.User != "nobody" {
					// Nanny users is always act as root
					sshUsage.User = "root"
				}

				if err != nil {
					return nil, fmt.Errorf("unable to parse field%d: %v", i, err)
				}
			}

			out = append(out, sshUsage)
		}
	}

	return out, sc.Error()
}
