package yp

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/yp/go/yp"
)

const (
	selectLimit = 500
	permsLimit  = 500
)

type Client struct {
	ypc *yp.Client
	log log.Logger
}

func NewClient(cluster string, opts ...Option) (*Client, error) {
	var l log.Logger = &nop.Logger{}
	var ypOpts []yp.Option
	for _, opt := range opts {
		switch o := opt.(type) {
		case AuthTokenOption:
			ypOpts = append(ypOpts, yp.WithAuthToken(o.token))
		case LoggerOption:
			ypOpts = append(ypOpts, yp.WithLogger(o.log))
			l = o.log
		default:
			return nil, fmt.Errorf("unsupported option: %T", opt)
		}
	}

	ypc, err := yp.NewClient(cluster, ypOpts...)
	if err != nil {
		return nil, err
	}

	return &Client{
		ypc: ypc,
		log: l,
	}, nil
}

func (c *Client) ListStages(ctx context.Context) ([]Stage, error) {
	var out []Stage
	var continuationToken string
	c.log.Info("list stages")
	total := 0
	for {
		stages, err := c.ypc.SelectStages(ctx, yp.SelectStagesRequest{
			Format:            yp.PayloadFormatYson,
			Limit:             selectLimit,
			ContinuationToken: continuationToken,
			Filter:            "[/meta/account_id] != 'tmp'",
			Selectors: []string{
				"/meta/id",
				"/meta/project_id",
			},
		})
		if err != nil {
			return nil, fmt.Errorf("failed to request stages: %s", err)
		}

		for stages.Next() {
			var stageID, projectID string
			if err := stages.Fill(&stageID, &projectID); err != nil {
				c.log.Error("unable to fill stage info", log.Error(err))
				continue
			}

			out = append(out, Stage{
				ID:      stageID,
				Project: projectID,
			})
		}

		continuationToken = stages.ContinuationToken()
		if continuationToken == "" {
			break
		}

		if stages.Count() < selectLimit {
			break
		}

		total += stages.Count()
		c.log.Infof("processed %d stages", total)
	}

	c.log.Info("fill stage permissions")
	if err := c.fillStagePermissions(ctx, out); err != nil {
		return nil, fmt.Errorf("unable to fill stage SSH users: %w", err)
	}

	c.log.Info("stage listed")
	return out, nil
}

func (c *Client) fillStagePermissions(ctx context.Context, stages []Stage) error {
	fetch := func(startIdx int, perms []yp.ObjectAccessAllowedForPermission) error {
		if len(perms) == 0 {
			return nil
		}

		rsp, err := c.ypc.ObjectAccessAllowedFor(ctx, yp.ObjectAccessAllowedForRequest{
			Permissions: perms,
		})
		if err != nil {
			return err
		}

		curIdx := startIdx
		for i, users := range rsp.Users {
			if (i+1)%3 == 0 {
				stages[curIdx].Writers = users.UserIDs
				curIdx++
			} else {
				stages[curIdx].SSHUsers = users.UserIDs
			}
		}
		return nil
	}

	var perms []yp.ObjectAccessAllowedForPermission
	var lastFetchedIdx, toFetch int
	for i, stage := range stages {
		perms = append(perms, yp.ObjectAccessAllowedForPermission{
			ObjectType:    yp.ObjectTypeStage,
			ObjectID:      stage.ID,
			Permission:    yp.AccessControlPermissionSshAccess,
			AttributePath: "/access/deploy/box/default/",
		})
		perms = append(perms, yp.ObjectAccessAllowedForPermission{
			ObjectType:    yp.ObjectTypeStage,
			ObjectID:      stage.ID,
			Permission:    yp.AccessControlPermissionRootSshAccess,
			AttributePath: "/access/deploy/box/default/",
		})
		perms = append(perms, yp.ObjectAccessAllowedForPermission{
			ObjectType: yp.ObjectTypeStage,
			ObjectID:   stage.ID,
			Permission: yp.AccessControlPermissionWrite,
		})

		toFetch++
		if toFetch < permsLimit {
			continue
		}

		c.log.Infof("processed %d stages", lastFetchedIdx+toFetch)
		if err := fetch(lastFetchedIdx, perms); err != nil {
			return err
		}

		toFetch = 0
		lastFetchedIdx = i + 1
		perms = perms[:0]
	}

	return fetch(lastFetchedIdx, perms)
}
