import sys
import os
import time
import argparse
import requests
import logging
import yaml
import json
import click

import ticket_parser2
import ticket_parser2.low_level as tp2
# from security.takeout import tvmknife
from library.python.vault_client.instances import Production as VaultClient
from library.python import resource


logging.basicConfig(level=logging.INFO)
logger = logging.getLogger('tester')

# "Production" Takeout: https://abc.yandex-team.ru/services/takeout_tester/resources/?show-resource=5136921
SELF_TVM_CLIENT_ID = int(os.environ.get('TAKEOUT_TVM_CLIENT_ID', 2010664))
SELF_TVM_SECRET = os.environ.get('TAKEOUT_TVM_SECRET', None)
SELF_TVM_SECRET_UUID = 'sec-01d4q7w715fz58v90ept2ra2f8'

# Fake Takeout: https://abc.yandex-team.ru/services/takeout_tester/resources/?show-resource=5137203
FAKE_TVM_CLIENT_ID = int(os.environ.get('TAKEOUT_FAKE_TVM_CLIENT_ID', 2010670))
FAKE_TVM_SECRET = os.environ.get('TAKEOUT_FAKE_TVM_SECRET', '4V8qIqy_1PFCwwOIH0cnWg')

# Some user in test passport
DEFAULT_VICTIM_UID = os.environ.get('TAKEOUT_VICTIM_UID', '4003515785')


def setup_tvm():
    global SELF_TVM_SECRET
    if SELF_TVM_SECRET:
        return

    try:
        client = VaultClient()
        head_version = client.get_version(SELF_TVM_SECRET_UUID)
        SELF_TVM_SECRET = head_version.get('value', {}).get('secret', None)
    finally:
        if not SELF_TVM_SECRET:
            logger.error(
                'failed to get tvm secret for client %d, please provide it in "TAKEOUT_TVM_CLIENT_ID" env or get access to secret "%s"',
                SELF_TVM_CLIENT_ID, SELF_TVM_SECRET_UUID
            )
            sys.exit(1)


def get_tvm_keyring_aliases(settings_type):
    cfg = json.loads(resource.find('tvm-keyring-%s.json' % settings_type))
    result = {}
    for dst in cfg.get('destinations', []):
        result[dst['alias']] = dst['client_id']
    return result


def get_takeout_upstreams(settings_type):
    tvm_aliases = get_tvm_keyring_aliases(settings_type)
    config = yaml.safe_load(resource.find('takeout-%s.yaml' % settings_type))
    services = config.get('services', None)
    if not services:
        return

    for name, service in services.items():
        # print(name, service)
        if not service.get('enabled', True):
            continue

        urls = service.get('urls', None)
        if not urls:
            continue

        upstream_path = urls.get('suffix_start', None) or urls.get('suffix_get', None)
        if not upstream_path:
            logger.warn('failed to construct upstream url, service (%s): %s', name, service)
            continue

        upstream_url = urls['base'] + upstream_path
        upstream_client_id = tvm_aliases.get(service.get('tvm_dst_alias'), None)
        yield upstream_url, upstream_client_id


def generate_tvm_ticket(dst_id, client_id=None, client_secret=None):
    if not client_id:
        client_id = SELF_TVM_CLIENT_ID

    if not client_secret:
        client_secret = SELF_TVM_SECRET

    tvm_keys = requests.get('https://tvm-api.yandex.net/2/keys?lib_version={version}'.format(version=ticket_parser2.__version__)).content
    sc = tp2.ServiceContext(client_id, client_secret, tvm_keys.decode())
    ts = str(int(time.time()))
    tickets = requests.post('https://tvm-api.yandex.net/2/ticket/', data={
        'grant_type': 'client_credentials',
        'src': client_id,
        'dst': dst_id,
        'ts': ts,
        'sign': sc.sign(ts, str(dst_id), ''),
    }).json()

    return tickets[str(dst_id)]['ticket']


def call_upstream(uid, url, ticket=None):
    headers = {}
    if ticket:
        headers['X-Ya-Service-Ticket'] = ticket
    return requests.post(
        url,
        headers=headers,
        data={
            'uid': uid,
            'unixtime': int(time.time())
        },
    )


def test_normal(uid, client_id, url):
    resp = call_upstream(
        uid=uid,
        url=url,
        ticket=generate_tvm_ticket(dst_id=client_id)
    )

    if resp.status_code != 200:
        logger.info('not 200 status code: %d, content:\n%s', resp.status_code, resp.text)
        return
    print(resp.json())


def test_wrong_dst(uid, client_id, url):
    resp = call_upstream(
        uid=uid,
        url=url,
        ticket=generate_tvm_ticket(dst_id=SELF_TVM_CLIENT_ID)
    )

    if resp.status_code != 200:
        logger.info('not 200 status code: %d, content:\n%s', resp.status_code, resp.text)
        return
    print(resp.json())


def test_wrong_src(uid, client_id, url):
    resp = call_upstream(
        uid=uid,
        url=url,
        ticket=generate_tvm_ticket(
            dst_id=client_id,
            client_id=FAKE_TVM_CLIENT_ID,
            client_secret=FAKE_TVM_SECRET
        )
    )

    if resp.status_code != 200:
        logger.info('not 200 status code: %d, content:\n%s', resp.status_code, resp.text)
        return
    print(resp.json())


def test_wo_tvm(uid, client_id, url):
    resp = call_upstream(
        uid=uid,
        url=url
    )

    if resp.status_code != 200:
        logger.info('not 200 status code: %d, content:\n%s', resp.status_code, resp.text)
        return
    print(resp.json())


def test_tvm_knife(uid, client_id, url):
    resp = call_upstream(
        uid=uid,
        url=url,
        ticket=tvmknife.create_service_ticket(src=SELF_TVM_CLIENT_ID, dst=client_id)
    )

    if resp.status_code != 200:
        logger.info('not 200 status code: %d, content:\n%s', resp.status_code, resp.text)
        return
    print(resp.json())


def run_tests(cases, uid, client_id, url):
    for case in cases:
        print('\n----------')
        print('call %s:' % case.__name__)
        try:
            case(uid=uid, client_id=client_id, url=url)
        except Exception as e:
            logger.error('failed %s', e)
        print('----------\n')


def run_all_tests(uid, client_id, url):
    run_tests(
        uid=uid,
        client_id=client_id,
        url=url,
        cases=[
            test_normal,
            test_wrong_dst,
            test_wrong_src,
            test_wo_tvm,
            # test_tvm_knife
        ],
    )


@click.group()
def cli():
    pass


@cli.command()
@click.option('--uid', type=int, required=True, default=DEFAULT_VICTIM_UID,
              help='victim UID')
@click.option('--client-id', type=int, required=True,
              help='target TVM application id')
@click.option('--url', type=str, required=True,
              help='target URL')
def single(uid, client_id, url):
    if uid <= 0:
        raise click.BadParameter('--uid must gather than 0')
    if client_id <= 0:
        raise click.BadParameter('--client-id must gather than 0')
    if not url:
        raise click.BadParameter('--url must a valid upstream url')

    setup_tvm()
    run_all_tests(uid=uid, client_id=client_id, url=url)


@cli.command()
@click.option('--uid', type=int, required=True, default=DEFAULT_VICTIM_UID,
              help='victim UID')
@click.option('env_type', '--env', type=click.Choice(['testing', 'production']),
              default='production', help='takeout settings type')
@click.option('list_only', '--list', is_flag=True,
              default=False, help='list services info only')
def whole(uid, env_type, list_only):
    if uid <= 0:
        raise click.BadParameter('--uid must gather than 0')

    upstreams = get_takeout_upstreams(env_type)
    if list_only:
        for upstream, client_id in upstreams:
            print('upstream: %s, client_id: %d' % (upstream, client_id))
    else:
        setup_tvm()
        for upstream, client_id in upstreams:
            print('check upstream: %s, client_id: %d' % (upstream, client_id))
            run_all_tests(uid=uid, client_id=client_id, url=upstream)


if __name__ == '__main__':
    cli()
