package app

import (
	"context"
	"html/template"
	"path/filepath"
	"time"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/security/tools/brobot/pkg/worker"
)

type App struct {
	echo   *echo.Echo
	worker *worker.Worker
	opts   Options
}

type Options struct {
	ProxyAddr  string
	AuthKey    string
	PublicPath string
}

func New(opts Options) *App {
	return &App{
		echo: echo.New(),
		opts: opts,
		worker: worker.NewWorker(
			worker.WithNWorkers(10),
			worker.WithMaxTasks(100),
			worker.WithProxyAddr(opts.ProxyAddr),
		),
	}
}

func (a *App) ListenAndServe(addr string) error {
	templates := template.Must(
		template.New("tmpl").
			Funcs(template.FuncMap{
				"formatTime": func(t *time.Time) string {
					return t.Format("02 Jan 06 15:04")
				},
			}).
			ParseGlob(filepath.Join(a.opts.PublicPath, "templates", "*.html")),
	)

	a.echo.Renderer = &Template{
		templates: templates,
	}

	a.echo.Use(middleware.RecoverWithConfig(middleware.RecoverConfig{
		Skipper:           middleware.DefaultSkipper,
		StackSize:         4 << 10, // 4 KB
		DisableStackAll:   true,
		DisablePrintStack: false,
	}))

	a.echo.GET("/", a.doIndex)
	a.echo.File("/favicon.ico", filepath.Join(a.opts.PublicPath, "static", "img", "favicon.ico"))
	a.echo.Static("/static", filepath.Join(a.opts.PublicPath, "static"))
	a.echo.GET("/ping", a.doPing)

	api := a.echo.Group(
		"/api",
		middleware.Logger(),
		middleware.KeyAuthWithConfig(middleware.KeyAuthConfig{
			KeyLookup: "header:X-Auth-Token",
			Validator: func(s string, _ echo.Context) (bool, error) {
				return s == a.opts.AuthKey, nil
			},
		}),
	)

	api.POST("/navigate", a.doNavigate)

	go a.worker.Loop()

	return a.echo.Start(addr)
}

func (a *App) Shutdown(ctx context.Context) error {
	if err := a.echo.Shutdown(ctx); err != nil {
		return err
	}

	return a.worker.Close(ctx)
}
