package worker

import (
	"context"
	"log"
	"sync"
	"time"

	"github.com/go-rod/rod"
	"github.com/go-rod/rod/lib/launcher"
	"github.com/go-rod/rod/lib/proto"
)

const (
	DefaultNWorkers = 10
	DefaultMaxTasks = 100
)

type Worker struct {
	tasks           chan Task
	done            chan struct{}
	log             *Log
	browserNameOnce sync.Once
	browserName     string
	proxyAddr       string
	nWorkers        int
	maxTasks        int
}

func WithNWorkers(n int) Option {
	return func(w *Worker) {
		w.nWorkers = n
	}
}

func WithMaxTasks(n int) Option {
	return func(w *Worker) {
		w.maxTasks = n
	}
}

func WithProxyAddr(addr string) Option {
	return func(w *Worker) {
		w.proxyAddr = addr
	}
}

type Option func(*Worker)

func NewWorker(opts ...Option) *Worker {
	w := &Worker{
		done:     make(chan struct{}),
		log:      NewLog(),
		nWorkers: DefaultNWorkers,
		maxTasks: DefaultMaxTasks,
	}

	for _, opt := range opts {
		opt(w)
	}

	w.tasks = make(chan Task, w.maxTasks)
	return w
}

func (w *Worker) NewTask(ctx context.Context, t Task) error {
	select {
	case w.tasks <- t:
		w.log.Log(t.id(), TaskStatusScheduled)
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
}

func (w *Worker) Loop() {
	defer close(w.done)

	var wg sync.WaitGroup
	wg.Add(w.nWorkers)

	for i := 0; i < w.nWorkers; i++ {
		go func() {
			defer wg.Done()
			for t := range w.tasks {
				w.doTask(t)
			}
		}()
	}

	wg.Wait()
}

func (w *Worker) Close(ctx context.Context) error {
	close(w.tasks)

	select {
	case <-w.done:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
}

func (w *Worker) Log() []LogItem {
	return w.log.Items()
}

func (w *Worker) BrowserName() string {
	w.browserNameOnce.Do(func() {
		browser, closeFn, err := w.newBrowser()
		if err != nil {
			w.browserName = "n/a"
			return
		}
		defer closeFn()

		v, err := proto.BrowserGetVersion{}.Call(browser)
		if err != nil {
			w.browserName = "n/a"
			return
		}

		w.browserName = v.Product
	})

	return w.browserName
}

func (w *Worker) doTask(t Task) {
	w.log.Log(t.id(), TaskStatusStarted)
	log.Printf("task started: %s\n", t.id())
	defer log.Printf("task done: %s\n", t.id())

	browser, closeFn, err := w.newBrowser()
	if err != nil {
		log.Printf("can't create new browser: %v\n", err)
		return
	}

	done := make(chan struct{})
	go func() {
		defer closeFn()
		defer close(done)

		if err := t.do(browser); err != nil {
			log.Printf("task failed: %v\n", err)
			return
		}
	}()

	timer := time.NewTimer(10 * time.Second)
	select {
	case <-timer.C:
		closeFn()
		w.log.Log(t.id(), TaskStatusKilled)
	case <-done:
		timer.Stop()
		w.log.Log(t.id(), TaskStatusCompleted)
	}
}

func (w *Worker) newBrowser() (*rod.Browser, func(), error) {
	l := launcher.New().
		Set("no-sandbox", "true").
		Set("disable-gpu", "true")

	if w.proxyAddr != "" {
		l = l.Proxy(w.proxyAddr)
	}

	url, err := l.Launch()
	if err != nil {
		return nil, nil, err
	}

	browser := rod.New().ControlURL(url)
	closeFn := func() {
		_ = browser.Close()
		l.Kill()
		l.Cleanup()
	}

	return browser, closeFn, browser.Connect()
}
