package commonformat

import (
	"encoding/binary"
	"errors"
	"io"
)

const (
	KindKv     = 1
	KindBytes  = 2
	KindUint64 = 3
	KindArray  = 4
	KindSecret = 5
)

var ErrCantSerialize = errors.New("can't serialize type")

type Packer struct {
	Writer io.Writer
}

func (p *Packer) Serialize(obj interface{}) error {
	_, err := p.Writer.Write([]byte{getKind(obj)})
	if err != nil {
		return err
	}
	err = p.pack(obj)
	if err != nil {
		return err
	}
	return nil
}

// TODO: pasrse errors correctly
func (p *Packer) pack(obj interface{}) error {
	switch o := obj.(type) {
	case map[string]interface{}:
		_, _ = p.Writer.Write([]byte{KindKv})
		_, _ = p.Writer.Write([]byte{uint8(len(o))})
		for key := range o {
			value := o[key]
			if value == nil {
				continue
			}
			size := make([]byte, 8)
			binary.LittleEndian.PutUint64(size, uint64(len(key)))
			_, _ = p.Writer.Write(size)
			_, _ = p.Writer.Write([]byte(key))
			if key == "secret" {
				_, _ = p.Writer.Write([]byte{KindSecret})
			} else {
				_, _ = p.Writer.Write([]byte{getKind(value)})
			}
			err := p.pack(value)
			if err != nil {
				return err
			}
		}
	case float64:
		number := make([]byte, 8)
		binary.LittleEndian.PutUint64(number, uint64(o))
		_, _ = p.Writer.Write(number)
	case string:
		size := make([]byte, 8)
		binary.LittleEndian.PutUint64(size, uint64(len(o)))
		_, _ = p.Writer.Write(size)
		_, _ = p.Writer.Write([]byte(o))
	case []interface{}:
		if len(o) == 0 {
			break
		}
		elem := o[0]
		switch elem.(type) {
		case float64:
			_, _ = p.Writer.Write([]byte{getKind(elem)})
			_, _ = p.Writer.Write([]byte{uint8(len(o))})
			for _, i := range o {
				err := p.pack(i)
				if err != nil {
					return err
				}
			}
		default:
			return ErrCantSerialize
		}
	default:
		return ErrCantSerialize
	}

	return nil
}

func getKind(value interface{}) uint8 {
	switch value.(type) {
	case map[string]interface{}:
		return KindArray
	case float64:
		return KindUint64
	case string:
		return KindBytes
	case []interface{}:
		return KindArray
	}
	return 0
}
