package cryptz

import (
	"bytes"
	"crypto/aes"
	"crypto/cipher"
	"crypto/rand"
	"errors"
)

var (
	ErrIncorrectPadding = errors.New("incorrect padding")
	ErrReadRand         = errors.New("read rand problem")
)

func padPKCS7(msg []byte) []byte {
	padBytes := 16 - len(msg)%16
	padding := bytes.Repeat([]byte{byte(padBytes)}, padBytes)
	return append(msg, padding...)
}

func EncryptAes(key []byte, msg []byte) ([]byte, error) {
	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}
	padMsg := padPKCS7(msg)
	size := 16 + len(padMsg)
	ciphertext := make([]byte, size)
	iv := ciphertext[:16]
	err = ReadRand(iv)
	if err != nil {
		return nil, err
	}
	cbc := cipher.NewCBCEncrypter(block, iv)
	cbc.CryptBlocks(ciphertext[16:], padMsg)
	return ciphertext, nil
}

func ReadRand(buf []byte) error {
	size := len(buf)
	n, err := rand.Read(buf)
	if err != nil {
		return err
	}
	if n != size {
		return ErrReadRand
	}
	return nil
}

func unpadPKCS7(msg []byte) ([]byte, error) {
	lastByte := msg[len(msg)-1]
	if lastByte > 16 {
		return nil, ErrIncorrectPadding
	}
	for i := 0; i < int(lastByte); i++ {
		if msg[len(msg)-1-i] != lastByte {
			return nil, ErrIncorrectPadding
		}
	}
	return msg[:len(msg)-int(lastByte)], nil
}

func decryptAes(key []byte, msg []byte) ([]byte, error) {
	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}
	iv := msg[:16]
	ciphertext := msg[16:]
	cbc := cipher.NewCBCDecrypter(block, iv)
	plaintext := make([]byte, len(ciphertext))
	cbc.CryptBlocks(plaintext, ciphertext)
	unpad, err := unpadPKCS7(plaintext)
	if err != nil {
		return nil, err
	}
	return unpad, nil
}
