package cryptz

import (
	"crypto/rand"
	"crypto/rsa"
	"crypto/sha256"
	"crypto/x509"
	"encoding/pem"
	"errors"
	"io/ioutil"
)

func EncryptEnvelope(rsaPub *rsa.PublicKey, msg []byte) ([]byte, error) {
	sessionKey := make([]byte, 32)
	err := ReadRand(sessionKey)
	if err != nil {
		return nil, err
	}
	enc, err := EncryptAes(sessionKey, msg)
	if err != nil {
		return nil, err
	}
	rng := rand.Reader
	sessKeyEnc, err := rsa.EncryptOAEP(sha256.New(), rng, rsaPub, sessionKey, nil)
	if err != nil {
		return nil, err
	}
	blob := make([]byte, len(sessKeyEnc)+len(enc))
	copy(blob[:], sessKeyEnc)
	copy(blob[len(sessKeyEnc):], enc)
	return blob, nil
}

func ReadPublic(filename string) (*rsa.PublicKey, error) {
	keyPem, err := ioutil.ReadFile(filename)
	if err != nil {
		return nil, err
	}
	block, _ := pem.Decode([]byte(keyPem))
	if block == nil {
		return nil, errors.New("can't decode as pem")
	}
	publicKey, err := x509.ParsePKIXPublicKey(block.Bytes)
	if err != nil {
		return nil, err
	}
	rsaKey, ok := publicKey.(*rsa.PublicKey)
	if !ok {
		return nil, errors.New("not public key")
	}
	return rsaKey, nil
}
