package maker

import (
	"bytes"
	"crypto/sha256"
	"encoding/binary"
	"encoding/json"

	"a.yandex-team.ru/security/tools/commongen/internal/commonformat"
	"a.yandex-team.ru/security/tools/commongen/internal/cryptz"
)

func MakeSecrets(configtext []byte, slugToSecret map[string][]byte) ([]byte, []byte, error) {
	// ilyaon: use json with reflection because I do not want strict types in packer.Serialize
	var config interface{}
	err := json.Unmarshal(configtext, &config)
	if err != nil {
		return nil, nil, err
	}

	initKey := make([]byte, 16)
	err = cryptz.ReadRand(initKey)
	if err != nil {
		return nil, nil, err
	}

	configMap := config.(map[string]interface{})
	slugs := configMap["slugs"]
	slugsMap := slugs.(map[string]interface{})

	for slug, entry := range slugsMap {
		entryMap := entry.(map[string]interface{})
		secret, err := processEntry(initKey, slug, entryMap)
		if err != nil {
			return nil, nil, err
		}
		slugToSecret[slug] = secret
	}

	var buf bytes.Buffer
	packer := commonformat.Packer{
		Writer: &buf,
	}
	err = packer.Serialize(config)
	if err != nil {
		return nil, nil, err
	}

	return buf.Bytes(), initKey, nil
}

func processEntry(initKey []byte, slug string, entry map[string]interface{}) ([]byte, error) {
	keyPath := entry["public_path"]
	keyPathString := keyPath.(string)
	public, err := cryptz.ReadPublic(keyPathString)
	if err != nil {
		return nil, err
	}
	secretBytes, err := buildSecret()
	if err != nil {
		return nil, err
	}

	// format {"1": "<-- base64 secret -->"} msg
	secret := make(map[string][]byte)
	secret["1"] = secretBytes
	msg, err := json.Marshal(secret)
	if err != nil {
		return nil, err
	}

	envSecret, err := cryptz.EncryptEnvelope(public, msg)
	if err != nil {
		return nil, err
	}
	digest := buildHash(slug, entry)
	secretAndHash := append(envSecret, digest...)
	finalSecret, err := cryptz.EncryptAes(initKey, secretAndHash)
	if err != nil {
		return nil, err
	}
	entry["secret"] = string(finalSecret)
	delete(entry, "public_path")

	return secretBytes, nil
}

func buildHash(slug string, entry map[string]interface{}) []byte {
	acls := entry["acls"]
	aclsMap := acls.(map[string]interface{})
	tvmSrc := aclsMap["tvm_srcs"]
	tvmSrcsArray := tvmSrc.([]interface{})

	var msg bytes.Buffer
	msg.Write([]byte(slug))

	for _, srcI := range tvmSrcsArray {
		src := srcI.(float64)
		_ = binary.Write(&msg, binary.LittleEndian, uint64(src))
	}

	digest := sha256.Sum256(msg.Bytes())
	return digest[:]
}

func buildSecret() ([]byte, error) {
	key := make([]byte, 128)
	err := cryptz.ReadRand(key)
	if err != nil {
		return nil, err
	}
	return []byte(key), nil
}
