package main

import (
	"bytes"
	"crypto/sha256"
	"crypto/tls"
	"encoding/hex"
	"flag"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"os"
)

func main() {
	hashStrings := []string{
		"46dcf2dcd2246fa4fd3a2d6cd47a164c893696afb45dd9390c71242fa63faeff",
		"776f85331312474514d5a561e8ba06483813d13ff79fa8b4cc04d5af2b464643",
		"e3c3192646536bc4ffae6dd34324ef9dd8d3b16dfa13a24e13180ef14ba21bba",
	}

	client := &http.Client{}
	tlsConfig := &tls.Config{InsecureSkipVerify: true}
	client.Transport = &http.Transport{
		DialTLS: func(network, addr string) (net.Conn, error) {
			conn, err := tls.Dial(network, addr, tlsConfig)
			if err != nil {
				return conn, err
			}

			state := conn.ConnectionState()
			certs := state.PeerCertificates
			if len(certs) != len(hashStrings) {
				log.Fatal("incorrect length of cert chain")
			}
			for i, cert := range state.PeerCertificates {
				digest := sha256.Sum256(cert.Raw)
				hash := hex.EncodeToString(digest[:])
				if hashStrings[i] != string(hash) {
					log.Fatal("cert digest not equal")
				}
			}

			return conn, nil
		},
	}

	host := flag.String("host", "", "Host to init.")
	initKeyPath := flag.String("init", "", "Path to init file")

	flag.Parse()
	if len(*host) == 0 || len(*initKeyPath) == 0 {
		flag.Usage()
		os.Exit(1)
	}

	initKeyHex, err := ioutil.ReadFile(*initKeyPath)
	if err != nil {
		log.Fatal(err)
	}

	initKeyHex = bytes.ReplaceAll(initKeyHex, []byte("\n"), []byte(""))

	initKey, err := hex.DecodeString(string(initKeyHex))
	if err != nil {
		log.Fatal(err)
	}

	buf := bytes.NewReader(initKey)
	resp, err := client.Post("https://"+*host+"/init", "application/octet-stream", buf)
	if err != nil {
		log.Fatal(err)
	}

	log.Println(resp)
}
