package childlog

import (
	"os"
	"sync"

	"a.yandex-team.ru/security/libs/go/simplelog"
)

type Writer struct {
	lock     sync.Mutex
	filename string
	fp       *os.File
	size     int
	maxSize  int
}

func New(filename string, maxSize int) (*Writer, error) {
	w := &Writer{
		filename: filename,
		maxSize:  maxSize,
	}

	err := w.rotateFile()
	if err != nil {
		return nil, err
	}
	return w, nil
}

func (w *Writer) Write(output []byte) (int, error) {
	w.lock.Lock()
	size, err := w.fp.Write(output)
	w.size += size
	needRotate := w.size >= w.maxSize
	w.lock.Unlock()

	if !needRotate {
		return size, err
	}

	if err := w.rotate(); err != nil {
		simplelog.Error("failed to rotate log", "filename", w.filename, "err", err)
	}

	return size, err
}

func (w *Writer) WriteLn(output []byte) (int, error) {
	return w.Write(append(output, '\n'))
}

func (w *Writer) rotate() error {
	w.lock.Lock()
	defer w.lock.Unlock()

	if w.size < w.maxSize {
		return nil
	}

	w.size = 0
	if w.fp != nil {
		err := w.fp.Close()
		w.fp = nil
		if err != nil {
			return err
		}
	}

	return w.rotateFile()
}

func (w *Writer) rotateFile() error {
	_, err := os.Stat(w.filename)
	if err == nil {
		err = os.Rename(w.filename, w.filename+".old")
		if err != nil {
			simplelog.Error("failed to rename old log", "old", w.filename, "new", w.filename+".old", "err", err)
		}
	}

	w.fp, err = os.Create(w.filename)
	return err
}

func (w *Writer) Close() error {
	if w.fp == nil {
		return nil
	}

	return w.fp.Close()
}
