package commands

import (
	"errors"
	"fmt"
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/tools/ctr/internal/compression"
	"a.yandex-team.ru/security/tools/ctr/internal/encrypt"
)

var decryptCmd = &cobra.Command{
	Use:          "decrypt [flags] [/path/to/dir]",
	SilenceUsage: true,
	Short:        "decrypt archive into directory",
	RunE: func(_ *cobra.Command, args []string) error {
		if len(args) != 1 {
			return errors.New("accept exactly one path")
		}

		return doDecrypt(args[0])
	},
}

func init() {
	flags := decryptCmd.PersistentFlags()
	flags.StringVar(&arcPath, "archive", arcPath, "input archive")
	rootCmd.AddCommand(decryptCmd)
}

func doDecrypt(target string) error {
	_, err := os.Stat(target)
	if err != nil && !os.IsNotExist(err) {
		return err
	}

	if err := os.MkdirAll(target, 0766); err != nil {
		return fmt.Errorf("can't create target directory: %w", err)
	}

	f, err := os.Open(arcPath)
	if err != nil {
		return fmt.Errorf("can't open archive file: %w", err)
	}
	defer func() { _ = f.Close() }()

	in, err := encrypt.NewReader(f, getKey())
	if err != nil {
		return fmt.Errorf("can't create encrypted reader: %w", err)
	}

	err = compression.UnCompress(target, in)
	if err != nil {
		return fmt.Errorf("uncompress failed: %w", err)
	}

	return nil
}
