package compression

import (
	"archive/tar"
	"compress/gzip"
	"fmt"
	"io"
	"os"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
)

func Compress(target string, out io.Writer, excludes ...string) error {
	gzw := gzip.NewWriter(out)
	defer func() { _ = gzw.Close() }()

	tw := tar.NewWriter(gzw)
	defer func() { _ = tw.Close() }()

	return CompressTo(target, tw, excludes...)
}

func CompressTo(target string, tw *tar.Writer, excludes ...string) error {
	skipThis := func(fi os.FileInfo) error {
		if fi.IsDir() {
			return filepath.SkipDir
		}
		return nil
	}

	isAcceptable := func(fm os.FileMode) bool {
		switch {
		case fm.IsRegular():
			return true
		case fm.IsDir():
			return true
		case fm&os.ModeSymlink != 0:
			return true
		default:
			return false
		}
	}

	prefix := cleanPath(target)
	return filepath.Walk(target, func(file string, fi os.FileInfo, err error) error {
		if err != nil {
			return err
		}

		if file == target {
			return nil
		}

		if !isAcceptable(fi.Mode()) {
			simplelog.Warn("skip unsupported entry", "path", file)
			return nil
		}

		basename := filepath.Base(file)
		for _, excl := range excludes {
			if basename == excl {
				return skipThis(fi)
			}
		}

		header, err := tar.FileInfoHeader(fi, file)
		if err != nil {
			return fmt.Errorf("can't create header (%s): %w", file, err)
		}

		header.Name = filepath.ToSlash(strings.TrimPrefix(file, prefix))
		if err := tw.WriteHeader(header); err != nil {
			return fmt.Errorf("can't write header (%s): %w", file, err)
		}

		for _, e := range emptyDirs {
			if header.Name == e {
				return skipThis(fi)
			}
		}

		if !fi.Mode().IsRegular() {
			return nil
		}

		f, err := os.Open(file)
		if err != nil {
			return err
		}
		defer func() { _ = f.Close() }()

		if _, err := io.Copy(tw, f); err != nil {
			return fmt.Errorf("copy failed (%s): %w", file, err)
		}
		return nil
	})
}
