package compression

import (
	"archive/tar"
	"compress/gzip"
	"fmt"
	"io"
	"os"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
)

func UnCompress(target string, src io.Reader) error {
	gzr, err := gzip.NewReader(src)
	if err != nil {
		return fmt.Errorf("can't create gzip reader: %w", err)
	}

	cTarget := cleanPath(target)
	tr := tar.NewReader(gzr)
	for {
		header, err := tr.Next()
		if err == io.EOF {
			break
		}

		if err != nil {
			return fmt.Errorf("can't get next entry: %w", err)
		}

		targetPath := filepath.Clean(filepath.Join(cTarget, header.Name))
		if !strings.HasPrefix(targetPath, cTarget) {
			return fmt.Errorf("invalid entry name: %s", header.Name)
		}

		dir := filepath.Dir(targetPath)
		mode := os.FileMode(header.Mode)

		switch header.Typeflag {
		case tar.TypeDir:
			err := os.MkdirAll(targetPath, mode)
			if err != nil && !os.IsExist(err) {
				return fmt.Errorf("can't create folder (%s): %w", targetPath, err)
			}

			// In some cases (e.g. directory was created by TypeReg), MkdirAll doesn't change the permissions, so run Chmod
			if err := os.Chmod(targetPath, mode); err != nil {
				return err
			}

		case tar.TypeReg:
			if err := prepareTarget(dir, targetPath); err != nil {
				return err
			}

			writeFile := func() error {
				f, err := os.OpenFile(targetPath, os.O_CREATE|os.O_WRONLY, mode)
				if err != nil {
					return err
				}
				defer func() { _ = f.Close() }()

				_, err = io.Copy(f, tr)
				return err
			}

			if err := writeFile(); err != nil {
				return fmt.Errorf("can't write file (%s): %w", targetPath, err)
			}

		case tar.TypeSymlink:
			if err := prepareTarget(dir, targetPath); err != nil {
				return err
			}

			if err := os.Symlink(header.Linkname, targetPath); err != nil {
				return fmt.Errorf("can't create symlink %s -> %s: %w", header.Linkname, targetPath, err)
			}

		default:
			simplelog.Warn("ignore unsupported entry type", "type", header.Typeflag)
		}
	}

	return nil
}

func prepareTarget(dir, target string) error {
	// It's possible a file is in the tar before it's directory.
	if _, err := os.Stat(dir); os.IsNotExist(err) {
		if err := os.MkdirAll(dir, 0755); err != nil {
			return fmt.Errorf("can't create target file (%s) directory: %w", target, err)
		}
	}

	// Check if something already exists at path (symlinks etc.)
	// If so, delete it
	if _, err := os.Lstat(target); !os.IsNotExist(err) {
		if err := os.RemoveAll(target); err != nil {
			return fmt.Errorf("can't remove %s to make way for new file", target)
		}
	}

	return nil
}
