package starter

import (
	"fmt"
	"os"
	"os/signal"
	"path/filepath"
	"strings"
	"syscall"

	"github.com/go-cmd/cmd"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/tools/ctr/internal/childlog"
)

type Env struct {
	Command string
	Args    []string
	LogDir  string
	AppName string
}

const (
	UnknownStatus = -1
	maxLogSize    = 1024 * 1024
)

var excludedEnv = []string{
	"CTR_KEY",
}

var passedSignals = []os.Signal{
	syscall.SIGINT,
	syscall.SIGKILL,
	syscall.SIGQUIT,
	syscall.SIGTERM,
	syscall.SIGHUP,
	syscall.SIGUSR1,
	syscall.SIGUSR2,
}

func Start(env Env) (int, error) {
	if err := os.MkdirAll(env.LogDir, 0766); err != nil {
		return UnknownStatus, fmt.Errorf("can't create logs dir: %w", err)
	}

	stdout, err := childlog.New(filepath.Join(env.LogDir, fmt.Sprintf("%s.stdout", env.AppName)), maxLogSize)
	if err != nil {
		return UnknownStatus, fmt.Errorf("can't create stdout log: %w", err)
	}
	defer func() { _ = stdout.Close() }()

	stderr, err := childlog.New(filepath.Join(env.LogDir, fmt.Sprintf("%s.stderr", env.AppName)), maxLogSize)
	if err != nil {
		return UnknownStatus, fmt.Errorf("can't create stderr log: %w", err)
	}
	defer func() { _ = stderr.Close() }()

	cmdOptions := cmd.Options{
		Buffered:  false,
		Streaming: true,
	}
	child := cmd.NewCmdOptions(cmdOptions, env.Command, env.Args...)
	child.Env = newEnv()

	streamDone := make(chan struct{})
	go func() {
		defer close(streamDone)

		for {
			select {
			case line, ok := <-child.Stdout:
				if !ok {
					simplelog.Info("EOF stdout child reached")
					child.Stdout = nil
					break
				}

				_, err = stdout.WriteLn([]byte(line))
				if err != nil {
					simplelog.Error("can't write child stdout", "err", err)
				}

			case line, ok := <-child.Stderr:
				if !ok {
					simplelog.Info("EOF stderr child reached")
					child.Stderr = nil
					break
				}

				_, err = stderr.WriteLn([]byte(line))
				if err != nil {
					simplelog.Error("can't write child stderr", "err", err)
				}
			}

			if child.Stdout == nil && child.Stderr == nil {
				break
			}
		}
	}()

	statusCh := child.Start()
	simplelog.Info("child started")

	sigCh := make(chan os.Signal, 1)
	signal.Notify(sigCh, passedSignals...)

	for {
		select {
		case sig := <-sigCh:
			simplelog.Info("pass to child received signal", "signal", sig)
			_ = sendSignal(child, sig)

		case status := <-statusCh:
			simplelog.Info("child stopped, cleanup signals && wait stdout/stderr", "code", status.Exit)
			signal.Stop(sigCh)
			close(sigCh)

			// wait stdout/stderr
			<-streamDone
			if status.Error != nil {
				return UnknownStatus, status.Error
			}
			return status.Exit, nil
		}
	}
}

func newEnv() []string {
	env := os.Environ()
	out := make([]string, 0, len(env))
	for _, e := range env {
		skip := false
		for _, excl := range excludedEnv {
			if strings.HasPrefix(e, excl) {
				skip = true
				break
			}
		}

		if skip {
			continue
		}

		out = append(out, e)
	}
	return out
}

func sendSignal(child *cmd.Cmd, sig os.Signal) error {
	pid := child.Status().PID
	if pid <= 0 {
		return fmt.Errorf("can't find child pid: %d", pid)
	}

	process, err := os.FindProcess(pid)
	if err != nil {
		return fmt.Errorf("can't find child process (%d): %w", pid, err)
	}

	return process.Signal(sig)
}
