package commands

import (
	"fmt"
	"io/ioutil"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/tools/dizzy/internal/config"
	"a.yandex-team.ru/security/tools/dizzy/internal/crt"
	"a.yandex-team.ru/security/tools/dizzy/internal/oauth"
	"a.yandex-team.ru/security/tools/dizzy/internal/sshkey"
	"a.yandex-team.ru/security/tools/dizzy/internal/staff"
)

var certCmd = &cobra.Command{
	Use:   "crt [flags] /ssh/key/path",
	Short: "Exchange the SSH key for OAuth token",
	RunE:  runCertCmd,
}

func init() {
	RootCmd.AddCommand(certCmd)
}

func runCertCmd(cmd *cobra.Command, args []string) error {
	fingerprint, err := sshkey.Fingerptint(config.SSHKey)
	if err != nil {
		return err
	}

	fmt.Printf("SSHKey fingerprint: %s\n", fingerprint)
	logins, err := staff.LoginsByKey(fingerprint)
	if err != nil {
		return err
	}

	if len(logins) == 0 {
		fmt.Println("SSH key not used by anyone")
		return nil
	}

	for _, login := range logins {
		logger := simplelog.Child("login", login)

		token, err := oauth.GetToken(config.SSHKey, login)
		if err != nil {
			logger.Warn("Failed to get OAuth token", "err", err.Error())
			continue
		}

		keyPEM, csrPEM, pubPEM, err := crt.GenCertificates(token, login)
		if err != nil {
			logger.Warn("Failed to generate&download certificate", "err", err.Error())
			continue
		}

		pubPath := fmt.Sprintf("%s.crt", login)
		keyPath := fmt.Sprintf("%s.key", login)
		csrPath := fmt.Sprintf("%s.csr", login)
		if err := ioutil.WriteFile(csrPath, csrPEM, 0644); err != nil {
			logger.Warn("Failed to save certificate", "path", csrPath, "err", err.Error())
			continue
		}
		if err := ioutil.WriteFile(keyPath, keyPEM, 0644); err != nil {
			logger.Warn("Failed to save certificate", "path", csrPath, "err", err.Error())
			continue
		}
		if err := ioutil.WriteFile(pubPath, pubPEM, 0644); err != nil {
			logger.Warn("Failed to save certificate", "path", csrPath, "err", err.Error())
			continue
		}

		fmt.Printf("Users %s certificate: \n", login)
		fmt.Printf("\t%s\n", keyPath)
		fmt.Printf("\t%s\n", csrPath)
		fmt.Printf("\t%s\n", pubPath)
	}
	return nil
}
