package commands

import (
	"fmt"
	"strings"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/tools/dizzy/internal/config"
	"a.yandex-team.ru/security/tools/dizzy/internal/idm"
	"a.yandex-team.ru/security/tools/dizzy/internal/sshkey"
	"a.yandex-team.ru/security/tools/dizzy/internal/staff"
)

var idmCmd = &cobra.Command{
	Use:   "idm [flags] /ssh/key/path",
	Short: "list ssh keys permissions",
	RunE:  runIdmCmd,
}

var system string

func init() {
	idmCmd.Flags().StringVar(&system, "system", "",
		"filter by system")

	RootCmd.AddCommand(idmCmd)
}

func runIdmCmd(cmd *cobra.Command, args []string) error {
	fingerprint, err := sshkey.Fingerptint(config.SSHKey)
	if err != nil {
		return err
	}

	fmt.Printf("SSHKey fingerprint: %s\n", fingerprint)
	logins, err := staff.LoginsByKey(fingerprint)
	if err != nil {
		return err
	}

	if len(logins) == 0 {
		fmt.Println("SSH key not used by anyone")
		return nil
	}

	fmt.Printf("Used by users: %s\n", strings.Join(logins, ", "))
	for _, login := range logins {
		logger := simplelog.Child("login", login)

		roles, err := idm.RolesByLogin(login, system)
		if err != nil {
			logger.Warn("failed to request IDM roles (result may be incomplete)", "err", err.Error())
		}

		fmt.Printf("'%s' IDM roles:\n", login)
		for _, role := range roles {
			if IsExcluded(role.Description) {
				continue
			}

			fmt.Println("\t" + role.Description)
		}
	}

	return nil
}
