package commands

import (
	"fmt"
	"strings"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/tools/dizzy/internal/conductor"
	"a.yandex-team.ru/security/tools/dizzy/internal/config"
	"a.yandex-team.ru/security/tools/dizzy/internal/idm"
	"a.yandex-team.ru/security/tools/dizzy/internal/sshkey"
	"a.yandex-team.ru/security/tools/dizzy/internal/staff"
)

var sshCmd = &cobra.Command{
	Use:   "ssh [flags] /ssh/key/path",
	Short: "list of hosts for SSH",
	RunE:  runSSHCmd,
}

func init() {
	RootCmd.AddCommand(sshCmd)
}

func runSSHCmd(cmd *cobra.Command, args []string) error {
	fingerprint, err := sshkey.Fingerptint(config.SSHKey)
	if err != nil {
		return err
	}

	fmt.Printf("SSHKey fingerprint: %s\n", fingerprint)
	logins, err := staff.LoginsByKey(fingerprint)
	if err != nil {
		return err
	}

	if len(logins) == 0 {
		fmt.Println("SSH key not used by anyone")
		return nil
	}

	fmt.Printf("Used by users: %s\n", strings.Join(logins, ", "))
	for _, login := range logins {
		logger := simplelog.Child("login", login)

		roles, err := idm.RolesByLogin(login, "cauth")
		if err != nil {
			logger.Warn("failed to request IDM roles (result may be incomplete)", "err", err.Error())
		}

		fmt.Printf("'%s' allowed SSH hosts:\n", login)
		for _, role := range roles {
			if role.Node["role"] != "ssh" {
				continue
			}
			dst := role.Node["dst"]
			if strings.HasPrefix(dst, "conductor.") {
				if IsExcluded(dst) {
					continue
				}

				group := dst[10:]
				hosts, err := conductor.HostsByGroup(group)
				if err != nil {
					continue
				}

				for _, h := range hosts {
					fmt.Printf("\t%s (%s)\n", h, group)
				}
			} else if strings.HasPrefix(dst, "cms.") {
				// TODO
			} else {
				fmt.Println("\t" + role.Node["dst"])
			}
		}
	}

	return nil
}
