package commands

import (
	"context"
	"errors"
	"io/ioutil"
	"log"
	"os"
	"strings"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/yandex/oauth"
	"a.yandex-team.ru/security/libs/go/sectools"
	"a.yandex-team.ru/security/tools/dizzy/internal/config"
)

var excludes = "conductor.infra_code conductor.disk_test_all conductor.sync_test_all conductor.yabs_bsdist"

var RootCmd = &cobra.Command{
	Use:          "dizzy [flags] /ssh/key/path",
	SilenceUsage: true,
	Short:        "SSH keys inspection",
	PersistentPreRunE: func(cmd *cobra.Command, args []string) error {
		if cmd.Name() == "version" {
			// OK
			return nil
		}

		if len(args) < 1 {
			return errors.New("SSH key needed")
		}

		key, err := ioutil.ReadFile(args[0])
		if err != nil {
			return err
		}
		config.KeyPath = args[0]
		config.SSHKey = key

		config.OAuthToken = os.Getenv("DIZZY_TOKEN")
		if config.OAuthToken == "" {
			// Try to get OAuth token from ssh
			config.OAuthToken, err = oauth.GetTokenBySSH(
				context.Background(),
				config.OAuthClientID, config.OAuthSecret,
			)
			if err != nil || config.OAuthToken == "" {
				log.Println("can't get oauth token from $DIZZY_TOKEN or exchange SSH key to token." +
					"Some functions may not work.")
			}
		}

		splitted := strings.Split(excludes, ",")
		config.Excludes = make([]string, len(splitted))
		for i, val := range splitted {
			config.Excludes[i] = strings.TrimSpace(val)
		}

		checkVersion()
		return nil
	},
}

func Execute() {
	if err := RootCmd.Execute(); err != nil {
		os.Exit(1)
	}
}

func init() {
	flags := RootCmd.PersistentFlags()
	flags.StringVar(&config.OAuthToken, "token", "",
		"IDM/Staff OAuth token")
	flags.StringVar(&excludes, "exclude", excludes,
		"coma-separated list to exclude")
}

func checkVersion() {
	ctx, cancel := context.WithTimeout(context.Background(), 1*time.Second)
	defer cancel()

	stClient := sectools.NewClient(config.ProductName)
	isLatest, latestVersion, err := stClient.IsLatestVersion(ctx, config.FullVersion())
	if !isLatest && err == nil {
		log.Println("----------")
		log.Printf("new version available: %s\n", latestVersion)
		log.Println("----------")
	}
}
