package crt

import (
	"bytes"
	"crypto/rand"
	"crypto/rsa"
	"crypto/x509"
	"crypto/x509/pkix"
	"encoding/asn1"
	"encoding/json"
	"encoding/pem"
	"fmt"
	"io/ioutil"
	"net/http"

	"a.yandex-team.ru/security/libs/go/yahttp"
	"a.yandex-team.ru/security/tools/dizzy/internal/utils"
)

func GenCertificates(token, login string) (keyPEM []byte, csrPEM []byte, pubPEM []byte, err error) {
	keyPEM, csrPEM, err = createCsr(login)
	if err != nil {
		return
	}

	pubURL, err := requestCrt(csrPEM, token, login)
	if err != nil {
		return
	}

	pubPEM, err = downloadCrt(token, pubURL)
	if err != nil {
		return
	}
	return
}

func createCsr(login string) ([]byte, []byte, error) {
	certKey, err := rsa.GenerateKey(rand.Reader, 2048)
	if err != nil {
		return nil, nil, err
	}

	template := &x509.CertificateRequest{
		SignatureAlgorithm: x509.SHA256WithRSA,
		PublicKeyAlgorithm: x509.RSA,
		PublicKey:          &certKey.PublicKey,
		Subject: pkix.Name{
			CommonName: fmt.Sprintf("%s@ld.yandex.ru", login),
			ExtraNames: []pkix.AttributeTypeAndValue{
				{
					Type:  asn1.ObjectIdentifier{1, 2, 840, 113549, 1, 9, 1},
					Value: fmt.Sprintf("%s@yandex-team.ru", login),
				},
			},
		},
		EmailAddresses: []string{fmt.Sprintf("%s@yandex-team.ru", login)},
	}
	csrDER, err := x509.CreateCertificateRequest(rand.Reader, template, certKey)
	if err != nil {
		return nil, nil, err
	}

	pemEncode := func(b []byte, t string) []byte {
		return pem.EncodeToMemory(&pem.Block{Bytes: b, Type: t})
	}

	keyPEM := pemEncode(x509.MarshalPKCS1PrivateKey(certKey), "RSA PRIVATE KEY")
	csrPEM := pemEncode(csrDER, "CERTIFICATE REQUEST")

	return keyPEM, csrPEM, nil
}

func requestCrt(csr []byte, token, login string) (uri string, err error) {
	b, err := json.Marshal(map[string]string{
		"type":        "linux-pc",
		"ca_name":     "InternalCA",
		"request":     string(csr),
		"pc_hostname": utils.RandHost(),
		"pc_os":       "linux",
		"pc_mac":      utils.RandMac(),
	})
	if err != nil {
		return
	}

	req, err := http.NewRequest("POST", "https://crt.yandex-team.ru/api/certificate/",
		bytes.NewBuffer(b))
	if err != nil {
		return
	}

	req.Header.Set("Content-Type", "application/json")
	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", token))
	resp, err := yahttp.DoRequest(req)
	if err != nil {
		return
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 201 {
		bb, _ := ioutil.ReadAll(resp.Body)
		err = fmt.Errorf("failed to create request crt (%d): %s", resp.StatusCode, string(bb))
		return
	}

	bb, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return
	}

	var crtResp struct {
		URL string `json:"download2"`
	}
	err = json.Unmarshal(bb, &crtResp)
	if err != nil {
		return
	}
	return crtResp.URL, nil
}

func downloadCrt(token, uri string) (res []byte, err error) {
	req, err := http.NewRequest("GET", uri, nil)
	if err != nil {
		return
	}

	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", token))
	resp, err := yahttp.DoRequest(req)
	if err != nil {
		return
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		bb, _ := ioutil.ReadAll(resp.Body)
		return nil, fmt.Errorf("failed to download crt (%d): %s", resp.StatusCode, string(bb))
	}

	bb, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return
	}

	return bb, nil
}
