package oauth

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"time"

	"a.yandex-team.ru/security/libs/go/yahttp"
	"a.yandex-team.ru/security/tools/dizzy/internal/config"
	"a.yandex-team.ru/security/tools/dizzy/internal/sshkey"
)

func GetToken(sshKey []byte, login string) (string, error) {
	ts := time.Now().Unix()
	toSign := fmt.Sprintf("%d%s%s", ts, config.OAuthClientID, login)
	sign, err := sshkey.SignPSS(sshKey, toSign)
	if err != nil {
		return "", err
	}

	token, err := request(config.OAuthClientID, config.OAuthSecret, login, sign, ts)
	if err != nil {
		return "", err
	}

	return token, nil
}

func request(clientID, clientSecret, login, sign string, ts int64) (token string, err error) {
	params := url.Values{
		"grant_type":    {"ssh_key"},
		"client_id":     {clientID},
		"client_secret": {clientSecret},
		"login":         {login},
		"ts":            {fmt.Sprintf("%d", ts)},
		"ssh_sign":      {sign},
	}

	req, err := http.NewRequest("POST", "https://oauth.yandex-team.ru/token",
		bytes.NewBufferString(params.Encode()))
	if err != nil {
		return
	}

	resp, err := yahttp.DoRequest(req)
	if err != nil {
		return
	}
	defer yahttp.GracefulClose(resp.Body)

	body, _ := ioutil.ReadAll(resp.Body)
	if resp.StatusCode != 200 {
		bb, _ := ioutil.ReadAll(resp.Body)
		err = fmt.Errorf("unexpected OAuth API response(%d): %s", resp.StatusCode, string(bb))
		return
	}

	var OAuthResponse struct {
		AccessToken string `json:"access_token"`
	}

	err = json.Unmarshal(body, &OAuthResponse)
	if err != nil {
		return
	}

	return OAuthResponse.AccessToken, nil
}
