package sshkey

import (
	"crypto"
	"crypto/ecdsa"
	"crypto/md5"
	"crypto/rand"
	"crypto/rsa"
	"crypto/x509"
	"encoding/base64"
	"encoding/pem"
	"errors"
	"fmt"

	"golang.org/x/crypto/ssh"
)

func Fingerptint(data []byte) (string, error) {
	signer, err := ssh.ParsePrivateKey(data)
	if err != nil {
		return "", err
	}

	hash := md5.Sum(signer.PublicKey().Marshal())
	fp := ""
	for i := 0; i < 16; i++ {
		if i > 0 {
			fp += ":"
		}
		fp += fmt.Sprintf("%02x", hash[i])
	}

	return fp, nil
}

func SignPSS(sshKey []byte, data string) (result string, err error) {
	block, _ := pem.Decode(sshKey)
	if block == nil {
		err = errors.New("failed to parse ssh key")
		return
	}

	privKey, err := parsePrivateKey(block.Bytes)
	if err != nil {
		return
	}

	if _, ok := privKey.(*rsa.PrivateKey); !ok {
		err = errors.New("supported only RSA keys")
		return
	}

	hashFn := crypto.SHA1
	hasher := hashFn.New()
	_, _ = hasher.Write([]byte(data))
	hash := hasher.Sum(nil)
	var signature []byte
	signature, err = rsa.SignPSS(rand.Reader, privKey.(*rsa.PrivateKey), hashFn, hash, nil)
	if err != nil {
		return
	}

	result = base64.URLEncoding.EncodeToString(signature)
	return
}

// Attempt to parse the given private key DER block. OpenSSL 0.9.8 generates
// PKCS#1 private keys by default, while OpenSSL 1.0.0 generates PKCS#8 keys.
// OpenSSL ecparam generates SEC1 EC private keys for ECDSA. We try all three.
func parsePrivateKey(der []byte) (crypto.PrivateKey, error) {
	if key, err := x509.ParsePKCS1PrivateKey(der); err == nil {
		return key, nil
	}
	if key, err := x509.ParsePKCS8PrivateKey(der); err == nil {
		switch key := key.(type) {
		case *rsa.PrivateKey, *ecdsa.PrivateKey:
			return key, nil
		default:
			return nil, errors.New("tls: found unknown private key type in PKCS#8 wrapping")
		}
	}
	if key, err := x509.ParseECPrivateKey(der); err == nil {
		return key, nil
	}

	return nil, errors.New("tls: failed to parse private key")
}
