package commands

import (
	"fmt"
	"io/fs"
	"os"
	"path/filepath"
	"sync"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yatool"
	"a.yandex-team.ru/security/libs/go/arc"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/tools/goat/internal/formatter"
)

const (
	fmtWorkers = 8
)

var fmtChanged bool

var fmtCmd = &cobra.Command{
	Use:   "fmt [flags] [target]",
	Short: "Format source code",
	RunE: func(_ *cobra.Command, args []string) error {
		if fmtChanged {
			return runArcFmt(args)
		}

		return runFsFmt(args)
	},
}

func init() {
	flags := fmtCmd.PersistentFlags()
	flags.BoolVar(&fmtChanged, "changed", false, "format only changed files")
	RootCmd.AddCommand(fmtCmd)
}

func runFsFmt(targets []string) error {
	if len(targets) == 0 {
		targets = []string{"."}
	}

	var (
		targetsWg sync.WaitGroup
		targetsCh = make(chan string, fmtWorkers)

		formattersWg sync.WaitGroup
		formatCh     = make(chan string, fmtWorkers)
	)

	targetsWg.Add(fmtWorkers)
	formattersWg.Add(fmtWorkers)
	for i := 0; i < fmtWorkers; i++ {
		go fsTargetWorker(targetsCh, formatCh, &targetsWg)
		go fmtWorker(formatCh, &formattersWg)
	}

	defer func() {
		close(targetsCh)
		targetsWg.Wait()

		close(formatCh)
		formattersWg.Wait()
	}()

	for _, target := range targets {
		if _, err := os.Stat(target); err != nil {
			return xerrors.Errorf("failed to resolve target %q: %w", target, err)
		}

		targetsCh <- target
	}

	return nil
}

func runArcFmt(targets []string) error {
	if len(targets) == 0 {
		targets = []string{"."}
	}

	var (
		formattersWg sync.WaitGroup
		formatCh     = make(chan string, fmtWorkers)
	)

	formattersWg.Add(fmtWorkers)
	for i := 0; i < fmtWorkers; i++ {
		go fmtWorker(formatCh, &formattersWg)
	}

	defer func() {
		close(formatCh)
		formattersWg.Wait()
	}()

	return resolveArcTargets(targets, formatCh)
}

func fsTargetWorker(targetsCh <-chan string, pathCh chan<- string, wg *sync.WaitGroup) {
	defer wg.Done()
	for target := range targetsCh {
		log := simplelog.Child("target", target)

		targetInfo, err := os.Stat(target)
		switch {
		case err != nil:
			log.Error("failed to resolve target", "err", err)
		case targetInfo.IsDir():
			err = filepath.WalkDir(target, func(osPathname string, d fs.DirEntry, err error) error {
				if err != nil {
					return err
				}

				if !formatter.IsGoDirEnt(d) {
					return nil
				}

				pathCh <- osPathname
				return nil
			})
			if err != nil {
				log.Error("failed to walk target", "err", err)
			}
		case formatter.IsGoFile(targetInfo):
			pathCh <- target
		}
	}
}

func resolveArcTargets(targets []string, pathCh chan<- string) error {
	arcRoot, err := yatool.FindArcadiaRoot(targets[0])
	if err != nil {
		return err
	}

	arcStatuses, err := arc.QueryStatusIn(targets...)
	if err != nil {
		return fmt.Errorf("can't query changes: %w", err)
	}

	collectFiles := func(statuses []arc.StatusEntry) {
		for _, status := range statuses {
			target := filepath.Join(arcRoot, status.Path)
			targetInfo, err := os.Stat(target)
			if err != nil {
				simplelog.Error("failed to resolve target", "path", status.Path, "err", err)
				continue
			}

			if !formatter.IsGoFile(targetInfo) {
				continue
			}

			pathCh <- target
		}
	}

	collectFiles(arcStatuses.Staged)
	collectFiles(arcStatuses.Changed)
	collectFiles(arcStatuses.Untracked)
	return nil
}

func fmtWorker(ch <-chan string, wg *sync.WaitGroup) {
	defer wg.Done()
	for path := range ch {
		if err := formatter.FormatFile(path); err != nil {
			simplelog.Error("failed to format file", "path", path, "err", err)
		}
	}
}
